<?php
/**
 * AnanyaPRO Theme Customizer
 *
 * @package AnanyaPRO
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Add postMessage support for site title and description for the Theme Customizer.
 *
 * @since 1.0.0
 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
 */
function ananyapro_customize_register( $wp_customize ) {
	$wp_customize->get_setting( 'blogname' )->transport         = 'postMessage';
	$wp_customize->get_setting( 'blogdescription' )->transport  = 'postMessage';

	//Register Custom Control - Range Control
	if ( method_exists( $wp_customize, 'register_control_type' ) ) {
		//Register - Range Control
		$wp_customize->register_control_type( 'Ananyapro_Range_Custom_Control' );
	}

	if ( isset( $wp_customize->selective_refresh ) ) {
		$wp_customize->selective_refresh->add_partial( 'blogname', array(
			'selector'        => '.site-title a',
			'render_callback' => 'ananyapro_customize_partial_blogname',
		) );
		$wp_customize->selective_refresh->add_partial( 'blogdescription', array(
			'selector'        => '.site-description',
			'render_callback' => 'ananyapro_customize_partial_blogdescription',
		) );

		//Button Selective refresh
		$wp_customize->selective_refresh->add_partial( 'ananyapro_header_title', array(
			'selector'        => '.hero-title',
			'render_callback' => 'ananyapro_print_hero_title',
		) );
		
		$wp_customize->selective_refresh->add_partial( 'ananyapro_footer_credit_text', array(
			'selector'        => '.site-info-container',
			'render_callback' => 'ananyapro_return_footer_credit_text',
		) );
	}

	// Add Theme Options Panel
	$wp_customize->add_panel( 'ananyapro_options_panel', array(
		'priority'       => 180,
		'capability'     => 'edit_theme_options',
		'theme_supports' => '',
		'title'          => esc_html__( 'Theme Other Options', 'ananyapro' ),
		'description'    => '',
	) );
	$wp_customize->add_panel( 'ananyapro_typography_panel', array(
		'priority'       => 42,
		'title'          => esc_html__( 'Typography', 'ananyapro' ),
		'description'    => '',
	) );
	$wp_customize->add_panel( 'ananyapro_frontpage_panel', array(
		'priority'       => 43,
		'title'          => esc_html__( 'Frontpage Sections', 'ananyapro' ),
		'description' 	 => esc_html__('Drag and Drop to Reorder', 'ananyapro') . '<img class="ananyapro-drag-spinner" src="' . admin_url('/images/spinner.gif') . '">',
	) );
}
add_action( 'customize_register', 'ananyapro_customize_register' );

/**
 * Render the site title for the selective refresh partial.
 *
 * @since 1.0.0
 * @return void
 */
function ananyapro_customize_partial_blogname() {
	bloginfo( 'name' );
}

/**
 * Render the site tagline for the selective refresh partial.
 *
 * @since 1.0.0
 * @return void
 */
function ananyapro_customize_partial_blogdescription() {
	bloginfo( 'description' );
}

/**
 * Binds JS handlers to make Theme Customizer preview reload changes asynchronously.
 */
function ananyapro_customize_preview_js() {
	wp_enqueue_script( 'ananyapro-customizer-js', get_template_directory_uri() . '/assets/js/customize-preview.js', array( 'customize-preview' ), '20151215', true );
}
add_action( 'customize_preview_init', 'ananyapro_customize_preview_js' );

/**
 * Enqueue script for custom customize control.
 * @since 1.0.0
 */
function ananyapro_customizer_pane_scripts() {	
	wp_enqueue_style( 'ananyapro-customizer-style',  get_template_directory_uri().'/assets/css/customizer-style.css' );
	wp_enqueue_script('ananyapro-customizer-script', get_template_directory_uri() . '/assets/js/customize-scripts.js', array(), false, true);
	wp_enqueue_script('ananyapro-customizer-controls', get_template_directory_uri() . '/assets/js/customize-controls.js', array(), false, true);
	//Enqueue fontawesome style
	wp_enqueue_style( 'ananyapro-fontawesome-style',  get_template_directory_uri().'/assets/font-awesome/css/all.min.css' );
	//Enqueue v4 compatibility fontawesome style 
	wp_enqueue_style( 'ananyapro-fontawesome-style',  get_template_directory_uri().'/assets/font-awesome/css/v4-shims.min.css' );
	wp_localize_script(	'ananyapro-customizer-script', 'ananyapro_ajax_object',
	  array( 'ajax_url'  => admin_url( 'admin-ajax.php' ),		    
	  		 'security'  => wp_create_nonce( 'ananyapro-section-security-nonce' )
	  )	
	);
}
add_action( 'customize_controls_enqueue_scripts', 'ananyapro_customizer_pane_scripts' );

/**
  * Saves the Fronpage Section Names in order.
  *
  * @since 1.0.0
  *
  */
function ananyapro_order_sections() {
	if ( ! isset( $_POST['ananyapro_secure_ajax_nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['ananyapro_secure_ajax_nonce'] ), 'ananyapro-section-security-nonce' ) ) {
			wp_die();
		}
    check_ajax_referer( 'ananyapro-section-security-nonce','ananyapro_secure_ajax_nonce');
    if( current_user_can( 'edit_theme_options' ) ) {
	    if ( isset($_POST['sections']) ) {
	        set_theme_mod( 'ananyapro_frontpage_sections', ananyapro_sanitize_text_array( $_POST['sections'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized 
	    }
	}
    wp_die();
}
add_action('wp_ajax_ananyapro_order_sections', 'ananyapro_order_sections');

/**
 * This function allows for a full blown array to get sanitized
 * properly, while sanitizing each individual value in a key -> value pair.
 *
 * @param    array    $input_arr        The address input.
 * @return   array    $sanitize_input_arr    The sanitized input.
 */
function ananyapro_sanitize_text_array( $input_arr ) {
    // Initialize the new array that will hold the sanitize values
	$sanitize_input_arr = array();

	// Loop through the input and sanitize each of the values
	foreach ( $input_arr as $key => $val ) {
		
		$sanitize_input_arr[ $key ] = ( isset( $input_arr[ $key ] ) ) ?
			sanitize_text_field(  wp_unslash( $val ) ) :
			'';
	}

	return $sanitize_input_arr;
}

if ( ! function_exists( 'ananyapro_header_style' ) ) :
	/**
	 * Styles the header image and text displayed on the blog.
	 *
	 * @see ananyapro_custom_header_setup().
	 * @since 1.0.0
	 */
	function ananyapro_header_style() {

		$defaults = ananyapro_get_defaults();
		$header_text_color = get_theme_mod( 'ananyapro_site_header_text_color', $defaults['site_header_text_color'] );
		$header_text_color_rgb = ananyapro_hex2rgb( $header_text_color );
		$header_border_color_rgba = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.3)', $header_text_color_rgb );
		$hero_btn_border_color_rgba = vsprintf( 'rgba( %1$s, %2$s, %3$s, 1)', $header_text_color_rgb );
		$header_image = get_header_image();
		if ( ! empty( $header_image ) ) {
			$header_width = get_custom_header()->width;
			$header_height = get_custom_header()->height;
		}
		
		$full_screen_header = get_theme_mod( 'ananyapro_full_screen_header_image_setting', true );
		$overlay_type = get_theme_mod( 'ananyapro_header_overlay_type', $defaults['header_overlay'] );
		$overlay_color = get_theme_mod( 'ananyapro_overlay_color', $defaults['overlay_color'] );
		$gradient_start_color = get_theme_mod( 'ananyapro_gradient_start_color', $defaults['gradient_start_color'] );
		$gradient_end_color = get_theme_mod( 'ananyapro_gradient_end_color', $defaults['gradient_end_color'] );
		$color_overlay_opacity = get_theme_mod( 'ananyapro_color_overlay_opacity', $defaults['overlay_opacity']);
		$gradient_overlay_opacity = get_theme_mod( 'ananyapro_gradient_overlay_opacity', $defaults['overlay_opacity']);
		?>
		<style type="text/css">

			<?php if( !ananyapro_display_site_title_tagline() ): ?>
				.site-title,
				.site-description {
					position: absolute;
					clip: rect(1px 1px 1px 1px); /* IE7 */
					clip: rect(1px, 1px, 1px, 1px);
				}
			<?php endif; ?>
			<?php if( $defaults['site_header_text_color'] != $header_text_color ) { ?>
				.site-title a, .site-title a:visited,.site-title a:hover,.site-description, .navbar-light .navbar-toggler {
					color: <?php echo esc_attr($header_text_color); ?>;
				}
				.over-hero-page-title .entry-title a, .over-hero-page-title .entry-title a:visited, .over-hero-page-title .entry-description a, .over-hero-page-title .entry-description a:visited, .over-hero-page-title .single-entry-meta a, .over-hero-page-title .single-entry-meta a:visited, .custom-header-other-page .blog-page-title {
					color: <?php echo esc_attr($header_text_color); ?>;
				}
				.over-hero-page-title .entry-title, .over-hero-page-title .entry-description, .over-hero-page-title .single-entry-meta {
					color: <?php echo esc_attr($header_text_color); ?>;
				}
				<?php if( class_exists( 'WooCommerce' ) ) { ?>
				.over-hero-page-title .woocommerce-products-header__title {
					color: <?php echo esc_attr($header_text_color); ?>;
				}
				<?php } ?>
				@media screen and (min-width: 768px) {
					#site-navigation .nav-menu-primary > li.menu-item > a,#site-navigation .nav-menu-primary > li.menu-item > a i, #site-navigation .nav-menu-primary > li.page_item > a,#site-navigation .nav-menu-primary > li.page_item > a i,.main-navigation a,.main-navigation a i,.header-social-menu li a,.header-social-menu li a i,.header-social-menu li #desktop-search-icon {
						color: <?php echo esc_attr($header_text_color); ?>;
					}
					#site-navigation .nav-menu-primary > li.menu-item > a:hover,#site-navigation .nav-menu-primary > li.menu-item > a i:hover, #site-navigation .nav-menu-primary > li.page_item > a:hover,#site-navigation .nav-menu-primary > li.page_item > a i:hover,.header-social-menu li a:hover,.header-social-menu li a i:hover,.header-social-menu li #desktop-search-icon:hover {
						color: <?php echo esc_attr($header_text_color); ?>;
						opacity: 0.7;
					}
				}
				@media screen and (min-width: 992px) {
					.header-social-menu ul {
						border-left: 1px solid <?php echo esc_attr($header_border_color_rgba); ?>;
					}
				}
				.custom-heading, .custom-heading .hero-title, .custom-heading .subheading, .custom-heading .hero-section-btn a, .custom-heading .hero-section-btn a:visited {
					color: <?php echo esc_attr($header_text_color); ?>;
				}
				.hero-btn-border-straight, .hero-btn-border-slight-round, .hero-btn-border-round {
					background: transparent !important;
					border-color: <?php echo esc_attr( $hero_btn_border_color_rgba ); ?>;
					color: <?php echo esc_attr( $header_text_color ); ?>;
				}
				.hero-btn-underline {
					background: transparent !important;
					border-bottom: 2px solid <?php echo esc_attr($header_text_color); ?>;
					color: <?php echo esc_attr($header_text_color); ?>;
					padding: 15px 0;
				}
				.home #site-navigation .container {
					border-bottom: 1px solid <?php echo esc_attr($header_border_color_rgba); ?>;
				}
				.hero-btn-fill-straight,.hero-btn-fill-slight-round,.hero-btn-fill-round {
					background: <?php echo esc_attr($header_text_color); ?>;
				}
				.hero-btn-border-straight,.hero-btn-border-slight-round,.hero-btn-border-round {
					border-color: <?php echo esc_attr($header_text_color); ?>;
					color: <?php echo esc_attr($header_text_color); ?>;
				}
				.hero-btn-underline {
					border-bottom-color: <?php echo esc_attr($header_text_color); ?>;
					color: <?php echo esc_attr($header_text_color); ?>;
				}
			<?php } ?>

			<?php if ( !$full_screen_header ) : ?>
				.has-header-image .custom-header,
				.has-header-video .custom-header {
					display: table;
					height: 400px;
					height: 80vh;
					width: 100%;
				}
				.admin-bar .has-header-image .custom-header, 
				.admin-bar .has-header-video .custom-header {
					height: calc(80vh - 32px);
				}
				@media screen and (min-width: 768px) {
					.custom-header-media {
						height: 600px;
						height: 80vh;
						max-height: 100%;
						overflow: hidden;
						position: relative;
					}
				}
			<?php endif; ?>

			<?php if( ( 'gradient-overlay' === $overlay_type ) && ( ( $gradient_start_color !== $defaults['gradient_start_color'] ) || ( $gradient_end_color !== $defaults['gradient_end_color'] ) ) ) : ?>
				.gradient-overlay .wp-custom-header:after {
					background: -moz-radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					background: -webkit-radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					background: radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=<?php echo esc_attr($gradient_start_color); ?>, endColorstr=<?php echo esc_attr( $gradient_end_color ); ?>,GradientType=1 ); 
				}
			<?php endif; ?>

			<?php if( 'gradient-overlay' === $overlay_type  &&   $gradient_overlay_opacity !== $defaults['overlay_opacity'] ) : ?>
				.gradient-overlay .wp-custom-header:after {
					opacity: <?php echo esc_attr( $gradient_overlay_opacity); ?>;
				}
			<?php endif; ?>
			
			<?php if( 'color-overlay' === $overlay_type && ( $overlay_color !== $defaults['overlay_color'] || $color_overlay_opacity !== $defaults['overlay_opacity'] ) ) :
					$overlay_color_rgb = ananyapro_hex2rgb( $overlay_color ); 
					$overlay_color_rgb[3] = esc_attr( $color_overlay_opacity );
					$overlay_color_rgba = vsprintf( 'rgba( %1$s, %2$s, %3$s, %4$s )', $overlay_color_rgb ); ?>
					.color-overlay .wp-custom-header:after {
						background: <?php echo esc_attr( $overlay_color_rgba ); ?>;
					}
			<?php endif; ?>

			<?php $total_no_sections = $defaults['total_no_of_each_type_section']; for( $sec_no=1; $sec_no <= $total_no_sections; $sec_no++ ) { $cta_sec_bg_type = get_theme_mod( 'ananyapro_cta_bg_type_sec'.$sec_no, $defaults['cta-bg-type'] ); if( 'gradient' === $cta_sec_bg_type ) { $gradient_start_color = get_theme_mod( 'ananyapro_cta_gradient_start_color_sec'.$sec_no, $defaults['gradient_start_color'] );$gradient_end_color = get_theme_mod( 'ananyapro_cta_gradient_end_color_sec'.$sec_no, $defaults['gradient_end_color'] ); ?>
					#cta-sec-<?php echo esc_attr($sec_no); ?> {
					background: -moz-radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					background: -webkit-radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					background: radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=<?php echo esc_attr($gradient_start_color); ?>, endColorstr=<?php echo esc_attr( $gradient_end_color ); ?>,GradientType=1 ); 
				}
			<?php } } ?>

			<?php $total_no_sections = $defaults['total_no_of_each_type_section']; for( $sec_no=1; $sec_no <= $total_no_sections; $sec_no++ ) { $cta_sec_bg_type = get_theme_mod( 'ananyapro_cta_bg_type_sec'.$sec_no, $defaults['cta-bg-type'] ); if( 'image-gradient' === $cta_sec_bg_type ) { $gradient_start_color = get_theme_mod( 'ananyapro_cta_gradient_start_color_sec'.$sec_no, $defaults['gradient_start_color'] );$gradient_end_color = get_theme_mod( 'ananyapro_cta_gradient_end_color_sec'.$sec_no, $defaults['gradient_end_color'] ); $cta_gradient_overlay_opacity = get_theme_mod( 'ananyapro_cta_gradient_overlay_opacity_sec'.$sec_no, $defaults['overlay_opacity'] );?>
					#cta-sec-<?php echo esc_attr($sec_no); ?> .cta-section-filter:before {
					background: -moz-radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					background: -webkit-radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					background: radial-gradient(left top,circle farthest-corner, <?php echo esc_attr($gradient_start_color); ?> 15%, <?php echo esc_attr( $gradient_end_color ); ?> 85%);
					filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=<?php echo esc_attr($gradient_start_color); ?>, endColorstr=<?php echo esc_attr( $gradient_end_color ); ?>,GradientType=1 );
					opacity: <?php echo esc_attr( $cta_gradient_overlay_opacity );?>;
				}
			<?php } } ?>
		</style>
		<?php
	}
endif;

/**
* This will output the custom WordPress settings to the live theme's WP head.
*
* Used by hook: 'wp_head'
*
* @see add_action('wp_head',$func)
* @since 1.0.0
*/
function ananyapro_accent_color_css() {
	$css1 = '';
	$defaults = ananyapro_get_defaults();
	$accent_color = get_theme_mod( 'ananyapro_accent_color', $defaults['accent_color'] );
	
	// Don't do anything if the current color is the default.
	if ( $accent_color !== $defaults['accent_color'] ) {
		
		$css1 ='.widget a:hover, .widget a:focus,a {
				color: %1$s;
			}
			.top-cat-link, .entry-meta .posted-on a {
				border-bottom: 2px solid %1$s;
			}
			.entry-header a:hover, blockquote:before,
			.footer-category-links i, .footer-tag-links i,.entry-edit-link a, .entry-edit-link a:visited, .entry-edit-link a:focus, .social-share a i, .social-share a:visited, .social-share a:focus i {
				color: %1$s;
			}
			.read-more .more-link:focus {
				outline-color: %1$s;
				outline-offset: 5px;
			}
			.more-btn-border-straight a, .more-btn-border-straight a:visited, .more-btn-border-straight a:hover, .more-btn-border-slight-round a, .more-btn-border-slight-round a:visited, .more-btn-border-slight-round a:hover, .more-btn-border-round a, .more-btn-border-round a:visited, .more-btn-border-round a:hover {
				border: 2px solid %1$s;
				background: #fff;
				color: %1$s;
			}
			.more-btn-underline a,.more-btn-underline a:visited,.more-btn-underline a:hover {
				border-bottom-color: %1$s;
				color: %1$s;
			}
			.more-btn-fill-straight a, .more-btn-fill-straight a:visited, .more-btn-fill-straight a:hover, .more-btn-fill-slight-round a, .more-btn-fill-slight-round a:visited, .more-btn-fill-slight-round a:hover, .more-btn-fill-round a, .more-btn-fill-round a:visited, .more-btn-fill-round a:hover {
				border: 2px solid %1$s;
				background: %1$s;
				color: #fff;
			}
			.pagination .current {
				background-color: %1$s;
				border: 1px solid %1$s;
				color: #fff;
			}
			#secondary .widget .widget-title {
				border-bottom: 2px solid %1$s;
				color: %1$s;
			}
			button, input[type=button], input[type=reset], input[type=submit], .woocommerce #payment #place_order, .woocommerce-page #payment #place_order, .woocommerce #respond input#submit.alt, .woocommerce a.button.alt, .woocommerce button.button.alt, .woocommerce input.button.alt, .woocommerce #respond input#submit, .woocommerce a.button, .woocommerce button.button, .woocommerce input.button, #infinite-handle span {
				border-radius: 3px;
				background: %1$s;
				color: #fff;
			}
			button:not(.dropdown-toggle):not(.feat-sec-btn):not(.hero-section-btn):not(.cta-sec-btn1)not(.cta-sec-btn2) {
				border-radius: 3px;
				background: %1$s;
				color: #fff;
			}
			.main-navigation ul ul a:hover,
			.main-navigation ul ul a:focus {
				color: %1$s;
			}
			
				.main-navigation ul a:hover,
				.main-navigation ul a:focus {
				    color: %1$s;
				}
				.main-navigation ul a:focus,
				.main-navigation .dropdown-toggle:focus,
				.main-navigation ul #navbar-search-box .search-field:focus {
				  outline: 1px solid %1$s;
				  outline-offset: -3px;
				  outline-color: %1$s;
				}
			
			.hero-btn-color-fill-straight,
			.hero-btn-color-fill-slight-round,
			.hero-btn-color-fill-round {
				background: %1$s;
				color: #fff;	
			}
			.hero-btn-color-border-straight,
			.hero-btn-color-border-slight-round,
			.hero-btn-color-border-round {
				border-color: %1$s;	
			}
			.scrolltop { 
				background-color: %1$s;
			}
			.btn-accent-color-straight,
			.btn-accent-color-slight-round,
			.btn-accent-color-round  {
				background: %1$s;
			}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css1, esc_attr( $accent_color ) ) );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_accent_color_css', 11 );

/**
 * Enqueues front-end CSS for the heading text color.
 *
 * @since 1.0.0
 */
function ananyapro_frontpage_sections_css() {
	$css3 = '';
	$defaults = ananyapro_get_defaults();
	$top_widget_section_bg_color = get_theme_mod( 'ananyapro_top_widget_section_bg_color', $defaults['section_bg_color'] );
	$bottom_widget_section_bg_color = get_theme_mod( 'ananyapro_bottom_widget_section_bg_color', $defaults['section_bg_color'] );
	
	$frontpage_button_bg_color = get_theme_mod( 'ananyapro_frontpage_button_bg_color', $defaults['accent_color'] );
	$frontpage_button_text_color = get_theme_mod( 'ananyapro_frontpage_button_text_color', $defaults['frontpage_button_text_color'] );
	$frontpage_button_border_color = get_theme_mod( 'ananyapro_frontpage_button_border_color', $defaults['accent_color'] );

	if( $top_widget_section_bg_color !== $defaults['section_bg_color'] ) {
		$widget_css = '
		#fp-widget-sec-top {
			background: %1$s;
		}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $widget_css, esc_attr( $top_widget_section_bg_color ) ) );
	}

	if( $bottom_widget_section_bg_color !== $defaults['section_bg_color'] ) {
		$widget_css = '
		#fp-widget-sec-bottom {
			background: %1$s;
		}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $widget_css, esc_attr( $bottom_widget_section_bg_color ) ) );
	}

	if( $bottom_widget_section_bg_color !== $defaults['section_bg_color'] ) {
		$widget_css = '
		.frontpage-widget-area {
			background: %1$s;
		}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $widget_css, esc_attr( $bottom_widget_section_bg_color ) ) );
	}

	if ( $frontpage_button_bg_color !== $defaults['accent_color'] || $frontpage_button_text_color !== $defaults['frontpage_button_text_color'] ) {
		$css3 = '.fp-btn-fill-straight,.fp-btn-fill-slight-round,.fp-btn-fill-round { 
					background: %1$s; 
					color: %2$s; 
					border-color: %3$s; }
				.dark .fp-btn-border-straight,.dark .fp-btn-border-slight-round,.dark .fp-btn-border-round {
					background: transparent;
					color: %2$s; 
					border-color: %3$s;
				}
				.fp-btn-underline {
					background: transparent; color: %2$s; border-bottom-color: %3$s;
				}
				.sec-divider-internal:before {
					border-color: %3$s;
				}
				.sec-divider:before {
					border-top: 3px solid %3$s;
				}
				.ananyapro-front-page .hero-btn-color-fill-straight,
				.ananyapro-front-page .hero-btn-color-fill-slight-round,
				.ananyapro-front-page .hero-btn-color-fill-round {
					background: %1$s;
					border-color: %3$s;
				}
				.ananyapro-front-page .hero-btn-color-border-straight,
				.ananyapro-front-page .hero-btn-color-border-slight-round,
				.ananyapro-front-page .hero-btn-color-border-round {
					border-color: %3$s;
				}
				.btn-fp-bg-color-straight,
				.btn-fp-bg-color-slight-round,
				.btn-fp-bg-color-round {
					background: %1$s;
				}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css3, esc_attr( $frontpage_button_bg_color ),esc_attr( $frontpage_button_text_color ), esc_attr( $frontpage_button_border_color ) ) );
	}

	// ================================================
	// == Add Different Section's Background Color CSS
	// ================================================
	for( $sec_no=1 ; $sec_no <= 2; $sec_no++ ) {
		$css_content = '';
		
		$content_sec_bg_color = get_theme_mod( 'ananyapro_content_section_bg_color_sec_'.$sec_no, $defaults['section_bg_color'] );
		if( $content_sec_bg_color !== $defaults['section_bg_color'] ) {
			$css_content .= '#feat-content-sec-'.$sec_no.' {
				background-color: %1$s;
			}';
			wp_add_inline_style( 'ananyapro-style', sprintf( $css_content, esc_attr( $content_sec_bg_color ) ) );
		}
		$css_category = '';
		$category_sec_bg_color = get_theme_mod( 'ananyapro_category_section_bg_color_sec_'.$sec_no, $defaults['section_bg_color'] );
		if( $category_sec_bg_color !== $defaults['section_bg_color'] ) {
			$css_category .= '#feat-cat-sec-'.$sec_no.' {
				background-color: %1$s;
			}';
			wp_add_inline_style( 'ananyapro-style', sprintf( $css_category, esc_attr( $category_sec_bg_color ) ) );
		}
		$css_page = '';
		$page_sec_bg_color = get_theme_mod( 'ananyapro_page_section_bg_color_sec_'.$sec_no, $defaults['section_bg_color'] );
		if( $page_sec_bg_color !== $defaults['section_bg_color'] ) {
			$css_page .= '#feat-pages-sec-'.$sec_no.' {
				background-color: %1$s;
			}';
			wp_add_inline_style( 'ananyapro-style', sprintf( $css_page, esc_attr( $page_sec_bg_color ) ) );
		}
		$css_post_sec = '';
		$latest_post_sec_bg = get_theme_mod( 'ananyapro_recent_post_sec_bg_color', $defaults['section_bg_color'] );
		if( $latest_post_sec_bg !== $defaults['section_bg_color'] ) {
			$css_post_sec .= '.latest-posts-section {
				background-color: %1$s;
			}';
			wp_add_inline_style( 'ananyapro-style', sprintf( $css_post_sec, esc_attr( $latest_post_sec_bg ) ) );
		}
		$css_post_sec = '';
		$showcase_cats_sec_bg = get_theme_mod( 'ananyapro_showcase_cat_sec_bg_color', $defaults['section_bg_color'] );
		if( $showcase_cats_sec_bg !== $defaults['section_bg_color'] ) {
			$css_post_sec .= '.showcase-categories-section {
				background-color: %1$s;
			}';
			wp_add_inline_style( 'ananyapro-style', sprintf( $css_post_sec, esc_attr( $showcase_cats_sec_bg ) ) );
		}
	}
	// ================================================
	// == Add Section's Button Color CSS
	// ================================================
	$sec_color_css = '';
	$sec_name_color = get_theme_mod( 'ananyapro_frontpage_sec_name_text_color', $defaults['frontpage_sec_name_text_color'] );
	if( $sec_name_color !== $defaults['frontpage_sec_name_text_color'] ) {
		$sec_color_css .= '.frontpage-section .sec-name {
			color: %1$s;
		}';
	}
	$sec_tagline_color = get_theme_mod( 'ananyapro_frontpage_sec_tagline_text_color', $defaults['frontpage_sec_tagline_text_color'] );
	if( $sec_tagline_color !== $defaults['frontpage_sec_tagline_text_color'] ) {
		$sec_color_css .= '.frontpage-section .sec-tagline{
			color: %2$s;
		}';
	}
	wp_add_inline_style( 'ananyapro-style', sprintf( $sec_color_css, esc_attr( $sec_name_color ), esc_attr( $sec_tagline_color ) ) );

}
add_action( 'wp_enqueue_scripts', 'ananyapro_frontpage_sections_css', 11 );

/**
 * Enqueues front-end CSS for the body text color.
 *
 * @since 1.0.0
 */
function ananyapro_body_text_color_css() {
	$css2 = '';
	$defaults = ananyapro_get_defaults();
	$body_text_color =  get_theme_mod( 'ananyapro_body_text_color', $defaults['body_text_color'] );
	$body_text_color_rgb = ananyapro_hex2rgb( $body_text_color );
	foreach ( $body_text_color_rgb as &$value) {
		$value = $value+10;
		if( $value > 255 ) { $value = 255; }
	}
	unset($value);//break the reference with the last element which was created by '&'.

	$sidebar_border_color_rgb = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.1)', $body_text_color_rgb );
	
	// Don't do anything if the current color is the default.
	if ( $body_text_color !== $defaults['body_text_color'] ) {
		$css2 = 'body,blockquote,cite,code,kbd,pre, .widget ul li a,.post-navigation a div, .post-navigation a div span, .pagination .page-numbers:not(.current),.entry-footer, .single-post-entry-footer {
				color: %1$s;
			}
			#secondary .widget ul li { border-bottom: 1px solid %2$s; }
			.entry-footer, .single-post-entry-footer {
				border-top: 1px solid %2$s;}
				.featured-page-section .page-read-more a, .featured-page-section .page-read-more a:visited{
					color: %1$s;
				}
				.latest-posts-section .post-read-more a, .latest-posts-section .post-read-more a:visited,.cat-read-more {
					color: %1$s;
				}
				.latest-posts-section .post-read-more a{
					border-bottom-color: %1$s;
				}
				.more-btn-border-gray-color a, .more-btn-border-gray-color a:visited, .more-btn-border-gray-color a:hover {
					border-color:  %1$s;
					color: %1$s;
				}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css2, esc_attr( $body_text_color ),esc_attr( $sidebar_border_color_rgb ) ) );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_body_text_color_css', 11 );

/**
 * Enqueues front-end CSS for the heading text color.
 *
 * @since 1.0.0
 */
function ananyapro_heading_text_color_css() {
	$css3 = '';
	$defaults = ananyapro_get_defaults();
	$heading_color = get_theme_mod( 'ananyapro_heading_text_color', $defaults['heading_text_color'] );
	
	// Don't do anything if the current color is the default.
	if ( $heading_color !== $defaults['heading_text_color'] ) {
		$css3 = 'h1,h2,h3,h4,h5,h6,.entry-title a, h1 a, h2 a, h3 a, h4 a, h5 a, h6 a,h1 a:visited, h2 a:visited, h3 a:visited, h4 a:visited, h5 a:visited, h6 a:visited {
				color: %1$s;
			}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css3, esc_attr( $heading_color ) ) );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_heading_text_color_css', 11 );

/**
 * Enqueues front-end CSS for the menu text color.
 *
 * @since 1.0.0
 */
function ananyapro_menu_item_textolor_css() {
	$defaults = ananyapro_get_defaults();
	
	$menu_item_text_color = get_theme_mod( 'ananyapro_menu_item_color', $defaults['menu_item_color'] );
	$menu_item_text_color_rgb = ananyapro_hex2rgb( $menu_item_text_color );
	
	// Don't do anything if the current color is the default.
	if ( $menu_item_text_color !== $defaults['menu_item_color'] ) {
		
		$css4 = '.no-header-image #site-navigation .nav-menu-primary > li.menu-item, .no-header-image #site-navigation .nav-menu-primary > li.menu-item > a, .no-header-image #site-navigation .nav-menu-primary > li.page_item, .no-header-image #site-navigation .nav-menu-primary > li.page_item > a, .no-header-image #site-navigation .header-social-menu ul, .no-header-image #site-navigation .header-social-menu li a i,
			.no-header-image #site-navigation .site-branding .site-title a, .no-header-image #site-navigation .site-branding .site-description,
			.no-header-image .navbar-light .navbar-toggler, .no-header-image .main-navigation ul ul a, .no-header-image .dropdown-toggle:hover, .no-header-image .dropdown-toggle:focus, .no-header-image .dropdown-toggle,.no-header-image .menu-item-cart-icon i, .no-header-image #site-navigation .nav-menu-primary > li.menu-item > a i{
					color: %1$s;
				}
			@media screen and (max-width: 991.99px) {
				.no-header-image .main-navigation ul #navbar-search-box .form-control {
					border: 1px solid %1$s;
					color: %1$s;
				}
			}
			@media screen and (min-width: 992px) {
				.no-header-image #site-navigation .header-social-menu ul {
					border-left-color: %1$s;
				} 
			}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css4, esc_attr( $menu_item_text_color ) ) );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_menu_item_textolor_css', 11 );

/**
 * Enqueues front-end CSS for the menu background color.
 *
 * @since 1.0.0
 */
function ananyapro_menu_bg_color_css() {
	$defaults = ananyapro_get_defaults();
	
	$menu_bg_color = get_theme_mod( 'ananyapro_menu_bg_color', $defaults['menu_bg_color'] );
	$menu_bg_color_rgb = ananyapro_hex2rgb( $menu_bg_color );
	$submenu_bg_color = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.7)', $menu_bg_color_rgb );
	$menu_item_text_color = get_theme_mod( 'ananyapro_menu_item_color','313131' );
	$menu_item_text_color_rgb = ananyapro_hex2rgb( $menu_item_text_color );
	$submenu_border_color = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.1)', $menu_item_text_color_rgb );
	
	// Don't do anything if the current color is the default.
	if ( $menu_bg_color !== $defaults['menu_bg_color'] ) {
		$css5 = '
			.no-header-image #site-navigation {
				background: %1$s;
			}
			.no-header-image .main-navigation ul,
			.no-header-image .main-navigation ul ul,
			.no-header-image #socialMenuResponsive,
			.no-header-image .dropdown-toggle,
			.no-header-image .dropdown-toggle:after {
				background: %1$s;
			}
			.no-header-image .main-navigation ul a {
				border-bottom-color: %2$s;
			}
			.no-header-image .dropdown-toggle:after {
				border-top-color:  %2$s;
				background: %1$s;
			}
			@media screen and (min-width: 992px) {
				.no-header-image .main-navigation ul ul li {
					border: none;
					border-top: 1px solid %1$s;
				}
				.no-header-image .main-navigation ul ul:after {
					border-color: %1$s; transparent;
				}
				.no-header-image .header-social-menu {
					border-bottom: none;
				}
				.no-header-image .main-navigation ul li > ul li:hover, .main-navigation ul li > ul li:focus {
					background: %3$s;
				}
			}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css5, esc_attr( $menu_bg_color ), esc_attr( $submenu_border_color) ,esc_attr( $submenu_bg_color ) ) );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_menu_bg_color_css', 11 );

/**
 * Enqueues front-end CSS for the post meta data text color.
 *
 * @since 1.0.0
 */
function ananyapro_post_meta_text_color_css() {
	$defaults = ananyapro_get_defaults();
	
	$meta_data_link_color = get_theme_mod( 'ananyapro_post_meta_color', $defaults['post_meta_color'] );
	$meta_data_link_color_rgb = ananyapro_hex2rgb( $meta_data_link_color );
	$meta_data_text_color= vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.7)', $meta_data_link_color_rgb );

	// Don't do anything if the current color is the default.
	if ( $meta_data_link_color !== $defaults['post_meta_color'] ) {
		$css6 = '.entry-meta a,.footer-category-links a, .footer-category-links a:visited, .footer-category-links a:focus, .footer-tag-links a, .footer-tag-links a:visited, .footer-tag-links a:focus, .entry-edit-link a, .entry-edit-link a:visited, .entry-edit-link a:focus, .social-share a, .social-share a:visited, .social-share a:focus, .single-entry-meta a, .single-entry-meta a:visited, .social-share {
				color: %1$s;
			}
			.entry-meta, .single-entry-meta { color: %2$s; }
			.latest-entry-meta .posted-on,.latest-entry-meta .posted-on a,.latest-entry-meta .first-cat,.latest-entry-meta .first-cat a,.cat-sec-content-wrap .posted-on a{
				color: %1$s;
			} ';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css6, esc_attr( $meta_data_link_color ), esc_attr( $meta_data_text_color ) ) );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_post_meta_text_color_css', 11 );

/**
 * Enqueues front-end CSS for the post meta data text color.
 *
 * @since 1.0.0
 */
function ananyapro_post_bg_color_css() {
	$defaults = ananyapro_get_defaults();
	
	$post_bg_color = get_theme_mod( 'ananyapro_post_bg_color', $defaults['post_bg_color'] );

	// Don't do anything if the current color is the default.
	if ( $post_bg_color !== $defaults['post_bg_color'] ) {
		$css7 = 'article, .post-navigation, .comments-area, .no-results,#secondary .widget, .pagination .page-numbers:not(.current) {
				background: %1$s;
			}
			.blog-page-header, .archive-page-header, .search-page-header {
				background: %1$s;
			}
			.single-product #main,.archive.woocommerce #main {
				background: %1$s;
			}
			.error404 #primary{
				background: %1$s;
			}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css7, esc_attr( $post_bg_color ) ) );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_post_bg_color_css', 11 );

/**
 * Enqueues front-end CSS for the footer background color.
 *
 * @since 1.0.0
 */
function ananyapro_footer_bg_color_css() {
	$defaults = ananyapro_get_defaults();
	$css8 = '';
	
	$footer_link_color = get_theme_mod( 'ananyapro_footer_link_color', $defaults['footer_link_color'] );
	$footer_link_color_rgb = ananyapro_hex2rgb( $footer_link_color );
	$footer_heading_color = get_theme_mod( 'ananyapro_footer_heading_color', $defaults['footer_text_color'] );
	$footer_text_color = get_theme_mod( 'ananyapro_footer_text_color', $defaults['footer_text_color'] );
	$footer_bg_color = get_theme_mod( 'ananyapro_footer_bg_color', $defaults['footer_bg_color'] );
	$footer_info_color_rgba = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.3 )', $footer_link_color_rgb );
	

	// Add CSS code for footer background color.
	if ( $footer_bg_color !== $defaults['footer_bg_color'] ) {
		$css8 = '.site-footer, #footer {
				background: %1$s;
			}';
		$css8 .= '.site-info {
					border-top: 1px solid %2$s;
				}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css8, esc_attr( $footer_bg_color ), esc_attr( $footer_info_color_rgba ) ) );
	}
	

	//Add css for footer text and link color
	$css9 = '';
	
	//Generate border color in footer from link color
	foreach ($footer_link_color_rgb as &$value) {
		$value = $value - 10;
		if( $value < 0 ){ $value = 0; }
	}
	unset($value); //break the reference with the last element which was created by '&'.
	
	$footer_border_color = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.7)', $footer_link_color_rgb );
	$footer_widget_border_color = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.2 )', $footer_link_color_rgb );

	unset($value); //break the reference with the last element which was created by '&'.
	// Don't do anything if the current color is the default.

	$css9 = '';

	if( $footer_text_color !== $defaults['footer_text_color'] || $footer_link_color !== $defaults['footer_link_color'] || $footer_heading_color !== $defaults['footer_text_color'] ) {

		if ( $footer_text_color !== $defaults['footer_text_color'] ) {
			$css9 = '.site-footer,  #footer,#footer .widget-title, #footer caption {
					color: %1$s;
				}';
		}
		if ( $footer_link_color !== $defaults['footer_link_color'] ) {
			$css9 .= '.site-footer a, #footer .widget a {
					color: %2$s;
				}
				#footer .widget-title:after {
					background-color: %3$s;
				}
				#footer .widget_calendar table td, #footer .widget_calendar table th {
					border: 1px solid %3$s;
				}';
		}
		if( $footer_heading_color !== $defaults['footer_text_color'] ) {
			$css9 .= '.site-footer h1, .site-footer h2,.site-footer h3,.site-footer h4,.site-footer h5,.site-footer h6,  #footer .widget-title {
					color: %4$s;
				}';
		}
		$css9 .= '#footer .widget_recent_comments ul li, #footer .widget_recent_entries ul li, #footer .widget_archive ul li, #footer .widget_categories ul li {
			border-bottom: 1px solid %5$s;
		}';
	}
	wp_add_inline_style( 'ananyapro-style', sprintf( $css9, esc_attr( $footer_text_color ), esc_attr( $footer_link_color ), esc_attr( $footer_border_color ), esc_attr( $footer_heading_color ), esc_attr($footer_widget_border_color) ) );

}
add_action( 'wp_enqueue_scripts', 'ananyapro_footer_bg_color_css', 11 );

/**
 * Convert HEX to RGB.
 *
 * @since 1.0.0
 *
 * @param string $color The original color, in 3- or 6-digit hexadecimal form.
 * @return array Array containing RGB (red, green, and blue) values for the given
 *               HEX code, empty array otherwise.
 */
function ananyapro_hex2rgb( $color ) {
	$color = trim( $color, '#' );

	if ( strlen( $color ) == 3 ) {
		$r = hexdec( substr( $color, 0, 1 ) . substr( $color, 0, 1 ) );
		$g = hexdec( substr( $color, 1, 1 ) . substr( $color, 1, 1 ) );
		$b = hexdec( substr( $color, 2, 1 ) . substr( $color, 2, 1 ) );
	} elseif ( strlen( $color ) == 6 ) {
		$r = hexdec( substr( $color, 0, 2 ) );
		$g = hexdec( substr( $color, 2, 2 ) );
		$b = hexdec( substr( $color, 4, 2 ) );
	} else {
		$r = hexdec( '12' );
		$g = hexdec( '34' );
		$b = hexdec( '56' );
	}

	return array(
		'red'   => $r,
		'green' => $g,
		'blue'  => $b,
	);
}

/**
 * Retrieve typography options.
 *
 * @since 1.0.0
 */
function ananyapro_print_typography_css() {
	$defaults = ananyapro_get_defaults();

	//boyd font family
	$body_font = ananyapro_retrieve_element_typography( 'ananyapro_body_font_select' );
	$heading_font = ananyapro_retrieve_element_typography( 'ananyapro_heading_font_select' );

	$blog_post_title_transform = get_theme_mod( 'ananyapro_blog_post_title_transform', $defaults['blog_post_title_transform'] );

	//body
	$body_d_font_size = get_theme_mod( 'ananyapro_body_font_size_desktop', $defaults['body_font_size_desktop'] );
	$body_m_font_size = get_theme_mod( 'ananyapro_body_font_size_mobile', $defaults['body_font_size_mobile'] );
	$body_font_line_height = get_theme_mod( 'ananyapro_body_line_height', $defaults['body_line_height'] );

	//post title
	$post_title_d_font_size = get_theme_mod( 'ananyapro_post_title_font_size_desktop', $defaults['post_title_font_size_desktop'] );
	$post_title_m_font_size = get_theme_mod( 'ananyapro_post_title_font_size_mobile', $defaults['post_title_font_size_mobile'] );
	$post_title_line_height = get_theme_mod( 'ananyapro_post_title_line_height', $defaults['post_title_line_height'] );
	$post_title_letter_spacing = get_theme_mod( 'ananyapro_post_title_letter_spacing', $defaults['post_title_letter_spacing'] );

	//hero font
	$hero_title_d_font_size = get_theme_mod( 'ananyapro_hero_title_font_size_desktop', $defaults['hero_title_font_size_desktop'] );
	$hero_title_m_font_size = get_theme_mod( 'ananyapro_hero_title_font_size_mobile', $defaults['hero_title_font_size_mobile'] );
	$hero_subtitle_d_font_size = get_theme_mod( 'ananyapro_hero_subtitle_font_size_desktop', $defaults['hero_subtitle_font_size_desktop'] );
	$hero_subtitle_m_font_size = get_theme_mod( 'ananyapro_hero_subtitle_font_size_mobile', $defaults['hero_subtitle_font_size_mobile'] );

	//menu font
	$menu_font_size = get_theme_mod( 'ananyapro_menu_font_size', $defaults['menu_font_size']);
	$menu_letter_spacing = get_theme_mod( 'ananyapro_menu_item_letter_spacing', $defaults['menu_item_letter_spacing'] );
	$menu_item_transform = get_theme_mod( 'ananyapro_menu_item_transform', $defaults['menu_item_transform'] );
	$menu_item_font_family = get_theme_mod( 'ananyapro_menu_item_font_family', $defaults['menu_item_font_family'] );

	$css10 = $css11 = '';
	$css10 = 'body,.subheading {
		font-family: %1$s;
		font-weight: %2$s; 
		text-transform: %3$s;
	}';
	wp_add_inline_style( 'ananyapro-style', sprintf( $css10, wp_kses_post( $body_font->family ) , esc_attr( $body_font->weight ),  esc_attr( $body_font->transformstyle ) ) ); //don't escape $body_font_family here

	
	$css11 = 'h1,h2,h3,h4,h5,h6 {
		font-family: %1$s;
		font-weight: %2$s;
		text-transform: %3$s;
	}
	blockquote{
		font-family: %1$s;
	}';

	wp_add_inline_style( 'ananyapro-style', sprintf( $css11, wp_kses_post( $heading_font->family ), esc_attr( $heading_font->weight ), esc_attr( $heading_font->transformstyle ) ) ); //don't escape $heading_font_family here

	if( $menu_item_font_family !== $defaults['menu_item_font_family'] ){
		$css = '.main-navigation li,
		main-navigation li a{
			font-family: %1$s;
		}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css, wp_kses_post( $heading_font->family ) ) );
	} 
	if( 'inherit' !== $blog_post_title_transform ) {
		$css12 = '.entry-title {
			text-transform: %1$s;
		}';
		wp_add_inline_style( 'ananyapro-style', sprintf( $css12, esc_attr( $blog_post_title_transform ) ) );
	}

	$css = '';
	//body font
	$css .= ananyapro_add_typography_css( $body_d_font_size, $defaults['body_font_size_desktop'], 'desktop', 'body', 'font-size', $body_d_font_size, 'px' );
	$css .= ananyapro_add_typography_css( $body_m_font_size, $defaults['body_font_size_mobile'], 'mobile', 'body', 'font-size', $body_m_font_size, 'px' );
	$css .= ananyapro_add_typography_css( $body_font_line_height, $defaults['body_line_height'], '', 'body', 'line-height', $body_font_line_height, '' );
	
	//hero title font size 
	$css .= ananyapro_add_typography_css( $hero_title_d_font_size, $defaults['hero_title_font_size_desktop'], 'desktop', '.custom-heading .hero-title', 'font-size', $hero_title_d_font_size, 'em' );
	$css .= ananyapro_add_typography_css( $hero_title_m_font_size, $defaults['hero_title_font_size_mobile'], 'mobile', '.custom-heading .hero-title', 'font-size', $hero_title_m_font_size, 'em' );
	//hero subheading fot size
	$css .= ananyapro_add_typography_css( $hero_subtitle_d_font_size, $defaults['hero_subtitle_font_size_desktop'], 'desktop', '.custom-heading .subheading', 'font-size', $hero_subtitle_d_font_size, 'px' );
	$css .= ananyapro_add_typography_css( $hero_subtitle_m_font_size, $defaults['hero_subtitle_font_size_mobile'], 'mobile', '.custom-heading .subheading', 'font-size', $hero_subtitle_m_font_size, 'px' );

	//post title font 
	$css .= ananyapro_add_typography_css( $post_title_d_font_size, $defaults['post_title_font_size_desktop'], 'desktop', '.blog .entry-title, .blog .blog-layout-list-1 article .entry-title, .archive .blog-layout-list-1 article .entry-title, .search .blog-layout-list-1 article .entry-title', 'font-size', $post_title_d_font_size, 'em' );
	$css .= ananyapro_add_typography_css( $post_title_m_font_size, $defaults['post_title_font_size_mobile'], 'mobile', '.blog .entry-title,.blog .blog-layout-list-1 article .entry-title, .archive .blog-layout-list-1 article .entry-title, .search .blog-layout-list-1 article .entry-title', 'font-size', $post_title_m_font_size, 'em' );
	$css .= ananyapro_add_typography_css( $post_title_line_height, $defaults['post_title_line_height'], '', '.blog .entry-title', 'line-height', $post_title_line_height, '' );
	$css .= ananyapro_add_typography_css( $post_title_letter_spacing, $defaults['post_title_letter_spacing'], '', '.blog .entry-title, .blog .entry-title a', 'letter-spacing', $post_title_letter_spacing, 'px' );

	//menu item
	$css .= ananyapro_add_typography_css( $menu_font_size, $defaults['menu_font_size'], '', '#navbarResponsive a', 'font-size', $menu_font_size, 'px' );
	$css .= ananyapro_add_typography_css( $menu_letter_spacing, $defaults['menu_item_letter_spacing'], '', '#navbarResponsive a', 'letter-spacing', $menu_letter_spacing, 'px' );
	$css .= ananyapro_add_typography_css( $menu_item_transform, $defaults['menu_item_transform'], '', '#nav-menu-primary a', 'text-transform', $menu_item_transform, '' );


	//heading line height and lette-spacing
	$heading_line_height = get_theme_mod( 'ananyapro_heading_line_height', $defaults['heading_line_height'] );
	$heading_letter_spacing = get_theme_mod( 'ananyapro_heading_letter_spacing', $defaults['heading_letter_spacing'] );
	$css .= ananyapro_add_typography_css( $heading_line_height, $defaults['heading_line_height'], '', 'h1,h2,h3,h4,h5,h6', 'line-height', $heading_line_height, '' );
	$css .= ananyapro_add_typography_css( $heading_letter_spacing, $defaults['post_title_letter_spacing'], '', 'h1,h2,h3,h4,h5,h6', 'letter-spacing', $heading_letter_spacing, 'px' );

	//heading font size
	for( $i = 1; $i <= 6; $i++ ) {
		$heading_font_size_d[$i-1] = get_theme_mod( 'ananyapro_heading'.$i.'_font_size_desktop', $defaults['heading'.$i.'_font_size_desktop'] );
		$heading_font_size_m[$i-1] = get_theme_mod( 'ananyapro_heading'.$i.'_font_size_mobile', $defaults['heading'.$i.'_font_size_mobile'] );
		$css .= ananyapro_add_typography_css( $heading_font_size_d[$i-1], $defaults['heading'.$i.'_font_size_desktop'], 'desktop', 'h'.$i, 'font-size', $heading_font_size_d[$i-1], 'em' );
		$css .= ananyapro_add_typography_css( $heading_font_size_m[$i-1], $defaults['heading'.$i.'_font_size_mobile'], 'mobile', 'h'.$i, 'font-size', $heading_font_size_m[$i-1], 'em' );
	}

	wp_add_inline_style( 'ananyapro-style', $css );
}
add_action( 'wp_enqueue_scripts', 'ananyapro_print_typography_css', 11 );

/**
 * Add typography CSS.
 *
 * @since 1.0.0
 */
function ananyapro_add_typography_css( $cntrl_value, $default, $device, $selector, $property, $pro_value, $unit ) {
	$css = '';
	if( 'desktop' === $device ) {
		$media_query_s =  '@media screen and  (min-width: 768px) {';
		$media_query_e =  '}';
	} else if( 'tablet' === $device ) {
		$media_query_s =  '@media screen and (min-width: 768px) and (max-width: 1024px) {';
		$media_query_e =  '}';
	} else if( 'mobile' === $device) {
		$media_query_s =  '@media screen and (max-width: 767px) {';
		$media_query_e =  '}';
	} else {
		$media_query_s =  '';
		$media_query_e =  '';
	}

	if( $cntrl_value !== $default ) {
		$css = '%1$s 
			%2$s { %3$s: %4$s%5$s }
		%6$s';
		$css = sprintf( $css, esc_attr( $media_query_s ),esc_attr( $selector ), esc_attr( $property ), esc_attr( $pro_value ), esc_attr( $unit ), esc_attr( $media_query_e) ) ;		
	}
	return $css;
}
/**
 * Add Frontpage CTA Section CSS.
 *
 * @since 1.0.0
 */
function ananyapro_frontpage_CTA_css() {
	$defaults = ananyapro_get_defaults();
	$total_no_sections = $defaults['total_no_of_each_type_section'];
	//button color scheme
		$frontpage_button_bg_color = get_theme_mod( 'ananyapro_frontpage_button_bg_color', $defaults['accent_color'] );
		$frontpage_button_text_color = get_theme_mod( 'ananyapro_frontpage_button_text_color', $defaults['frontpage_button_text_color'] );
		$frontpage_button_border_color = get_theme_mod( 'ananyapro_frontpage_button_border_color', $defaults['accent_color'] );

	for( $sec_no=1; $sec_no <= $total_no_sections; $sec_no++ ) {
		$cta_sec_bg_type = get_theme_mod( 'ananyapro_cta_bg_type_sec'.$sec_no, $defaults['cta-bg-type'] );
		$cta_bg_image = get_theme_mod( 'ananyapro_cta_bg_image_sec'.$sec_no, $defaults['cta_sec_bg_image'] );
		$cta_color_scheme = get_theme_mod( 'ananyapro_cta_text_color_scheme_sec'.$sec_no, $defaults['cta-color-scheme'] );
		$cta_bg_color = get_theme_mod( 'ananyapro_cta_bg_color_sec'.$sec_no, $defaults['cta_sec_bg'] );
		//$cta_bg_color_rgb = ananyapro_hex2rgb( $cta_bg_color );
		//$cta_bg_color_rgba = vsprintf( 'rgba( %1$s, %2$s, %3$s, 0.8)', $cta_bg_color_rgb );
		$cta_color_overlay_opacity = get_theme_mod( 'ananyapro_cta_color_overlay_opacity_sec'.$sec_no, $defaults['overlay_opacity'] );

		$css = '';
		$css .= '#cta-sec-'.$sec_no.' {
			background-color:%1$s;
		}';
		if( 'image' === $cta_sec_bg_type  ) {
			$css .= '#cta-sec-'.$sec_no.' {
				background-image: url( %2$s );
				background-size: cover;
			}';
		} 
		
		if( 'image-color' === $cta_sec_bg_type ) {
			$css .= '#cta-sec-'.$sec_no.' .cta-section-filter  {
				background-image: url( %2$s );
				background-size: cover;
			}
			#cta-sec-'.$sec_no.' .cta-section-filter:before  {
				background-color: %6$s;
				opacity: %7$s;
			}';
		}

		if( 'image-gradient' === $cta_sec_bg_type ) {
			$css .= '#cta-sec-'.$sec_no.' .cta-section-filter  {
				background-image: url( %2$s );
				background-size: cover;
			}';
		}

		if( 'dark' === $cta_color_scheme ) {
			$css .= '#cta-sec-'.$sec_no.' .cta-sec-btn1 {
				background: %5$s;
				border-color: %3$s;
				color: %4$s;
			}
			#cta-sec-'.$sec_no.'.dark  .cta-sec-btn1 a, #cta-sec-'.$sec_no.'.dark .cta-sec-btn1 a:visited{
				color: %4$s;
			}
			#cta-sec-'.$sec_no.'.dark  .cta-sec-btn1.fp-btn-underline a, #cta-sec-'.$sec_no.'.dark .cta-sec-btn1.fp-btn-underline a:visited{
				color: %3$s;
			}
			#cta-sec-'.$sec_no.' .cta-sec-btn2 {
				background: transparent;
				border-color:  %3$s;
				color: %3$s;
			}
			#cta-sec-'.$sec_no.'.dark  .cta-sec-btn2 a, #cta-sec-'.$sec_no.'.dark .cta-sec-btn2 a:visited {
				color: %3$s;
			}';
		}
		if( 'light' === $cta_color_scheme ) {
			$css .= '.cta-section.light .cta-sec-btn1 {
				background: %5$s;
				border-color: %3$s;
				color: #ffffff;
			}
			.cta-section.light .cta-sec-btn1 a, .cta-section.light .cta-sec-btn1 a:visited {
				color: %4$s;
			}
			.cta-section.light .cta-sec-btn1.fp-btn-border-straight,.cta-section.light .cta-sec-btn1.fp-btn-border-slight-round,.cta-section.light .cta-sec-btn1.fp-btn-border-round {
				background: transparent;
				border-color: %3$s;
				color: %3$s;
			}';
		}

		$css = sprintf( $css, esc_attr( $cta_bg_color ),esc_url( $cta_bg_image ), esc_attr( $frontpage_button_border_color ), esc_attr( $frontpage_button_text_color ), esc_attr( $frontpage_button_bg_color ), esc_attr( $cta_bg_color ), esc_attr($cta_color_overlay_opacity) ) ;	
		wp_add_inline_style( 'ananyapro-style', $css );
	}
}
add_action( 'wp_enqueue_scripts', 'ananyapro_frontpage_CTA_css', 11 );

/**
 * Retrieve typography options.
 *
 * @since 1.0.0
 */
function ananyapro_retrieve_element_typography( $customizer_control_name ) {
	$defaults = ananyapro_get_defaults();
	if('ananyapro_heading_font_select' === $customizer_control_name ) {
		$family = $defaults['heading_font_family'];
		$weight = $defaults['heading_font_weight'];
		$transformstyle = $defaults['heading_font_transformstyle'];
		$category = $defaults['heading_font_category'];
	} else {
		$family = $defaults['body_font_family'];
		$weight = $defaults['body_font_weight'];
		$transformstyle = $defaults['body_font_transformstyle'];
		$category = $defaults['body_font_category'];
	}
	$element_font_options = get_theme_mod( $customizer_control_name , json_encode(
		array(
			'family' => $family,
			'weight' => $weight,
			'transformstyle' => $transformstyle,
			'category' => $category,
		)
	));
	
	$element = json_decode( $element_font_options );
	
	$element->family = "'".esc_attr( $element->family )."',".esc_attr( $element->category );
	$element_is_italic = strpos( $element->weight, 'italic' );
	if( 'regular' == $element->weight ) {
		$element->weight = 'normal';
	} else {
		$element->weight = substr( $element->weight, 0, 3 );
	}
	if( $element_is_italic ) {
		$element->weight .= '; font-style: italic'; // Contact font weight and style css 
	} else {
		$element->weight .= '; font-style: normal';
	}
	return $element;
}
