<?php
/**
 * Custom template tags for this theme
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package AnanyaPRO
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'ananyapro_posted_on' ) ) :
	/**
	 * Prints HTML with meta information for the current post-date/time.
	 * @since 1.0.0
	 */
	function ananyapro_posted_on() {
		$posted_on_str = '';
		if( is_single() ) {
			$posted_on_str = esc_html__( 'On ', 'ananyapro' );
		} 
		$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
		if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
			$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
		}

		$time_string = sprintf( $time_string,
			esc_attr( get_the_date( DATE_W3C ) ),
			esc_html( get_the_date() ),
			esc_attr( get_the_modified_date( DATE_W3C ) ),
			esc_html( get_the_modified_date() )
		);

		$posted_on = sprintf(
			/* translators: %s: post date. */
			$posted_on_str.esc_html( '%s' ),
			'<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a>'
		);

		echo  '<span class="posted-on">' . $posted_on . '</span>' ;// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

	}
endif;

if ( ! function_exists( 'ananyapro_posted_by' ) ) :
	/**
	 * Prints HTML with meta information for the current author.
	 * @since 1.0.0
	 */
	function ananyapro_posted_by() {
		$by_str = esc_html__( 'By ', 'ananyapro' );
		if( is_single() ) {
			$by_str = esc_html__( 'By ', 'ananyapro' );
		}
		$byline = sprintf(
			/* translators: %s: post author. */
			$by_str.esc_html( '%s' ),
			'<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span>'
		);

		echo '<span class="byline"> ' . $byline . '</span>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

	}
endif;

if ( ! function_exists( 'ananyapro_entry_footer' ) ) :
	/**
	 * Prints HTML with meta information for the categories, tags and comments.
	 * @since 1.0.0
	 */
	function ananyapro_excerpt_entry_footer() { ?>
		<div class="row entry-footer">

			<div id="blog-cat-links" class="col-md-6 footer-category-links">
				<?php  ananyapro_post_categories(); ?>
			</div><!-- footer-category-links -->

			<div id="blog-social-share" class="col-md-6 social-share">
				<?php 
				echo wp_kses_post( ananyapro_social_sharing_buttons() );
				?>
			</div><!-- social-share -->

			<div id="blog-edit-link" class="col-md-12 entry-edit-link">
			<?php 
				edit_post_link(
					sprintf(
						wp_kses(
							/* translators: %s: Name of current post. Only visible to screen readers */
							__( 'Edit <span class="screen-reader-text">%s</span>', 'ananyapro' ),
							array(
								'span' => array(
									'class' => array(),
								),
							)
						),
						get_the_title()
					),
					'<span class="edit-link"><i class="fas fa-edit"></i>',
					'</span>'
				); 
			?>
			</div><!-- entry-edit-link -->

		</div><!-- row -->
		
		<?php
	}
endif;

if ( ! function_exists( 'ananyapro_post_excerpt_meta' ) ) :
	/**
	 * Prints HTML with meta information for the categories, tags and comments.
	 * @since 1.0.0
	 */
	function ananyapro_post_excerpt_meta() { 
		$hide_author = get_theme_mod( 'ananyapro_hide_blog_post_author', false );
		$hide_comments_link = get_theme_mod( 'ananyapro_hide_blog_post_comment_link', false );
		?>
		<div class="entry-meta">
			<?php if( !$hide_author ) { 
				ananyapro_posted_by(); } ?>
			<?php if( !$hide_comments_link ) { 
				ananyapro_post_comments_link(); } ?>
		</div><!-- entry-meta -->
	<?php
	}
endif;


if ( ! function_exists( 'ananyapro_single_post_meta' ) ) :
	/**
	 * Prints HTML with meta information for the categories, tags and comments.
	 * @since 1.0.0
	 */
	function ananyapro_single_post_meta() { 
		$hide_date = get_theme_mod( 'ananyapro_hide_single_post_date', false );
		$hide_author = get_theme_mod( 'ananyapro_hide_single_post_author', false );
		$hide_comments = get_theme_mod( 'ananyapro_hide_single_post_comment_link', false );
		?>
		<div class="single-entry-meta">
			<?php if( !$hide_author ) {
			ananyapro_posted_by();  } ?>
			<?php if( !$hide_date ) { 
			ananyapro_posted_on(); } ?>
			<?php if( !$hide_comments ) {
			ananyapro_post_comments_link(); }?>
		</div><!-- single-entry-meta -->
	<?php
	}
endif;

if ( ! function_exists( 'ananyapro_post_thumbnail' ) ) :
	/**
	 * Displays an optional post thumbnail.
	 *
	 * @since 1.0.0
	 */
	function ananyapro_post_thumbnail() {
		
		$header_image_only_on_front_page = get_theme_mod( 'ananyapro_use_header_image_only_on_front_page_setting', false );
		$hide_single_post_featured_image = get_theme_mod( 'ananyapro_hide_single_post_featured_image', false );
		$hide_blog_page_featured_image = get_theme_mod( 'ananyapro_hide_blog_post_featured_image', false );
		$page_header_uses_featured_image = get_theme_mod( 'ananyapro_single_page_featured_image_in_header', false );
		$post_header_uses_featured_image = get_theme_mod( 'ananyapro_single_post_featured_image_in_header', false );
		if ( post_password_required() || is_attachment() || ! has_post_thumbnail() ) {
			return;
		}

		if ( (is_single() && ( !$hide_single_post_featured_image  && !$post_header_uses_featured_image )) || (is_page() && !$page_header_uses_featured_image) || ((is_single() || is_page()) && ( !has_header_image() || $header_image_only_on_front_page) ) ) :
			?>
			<div class="post-thumbnail">
				<?php the_post_thumbnail(); ?>
			</div><!-- post-thumbnail -->

		<?php else : ?>
			<?php if( ( !is_singular() && !$hide_blog_page_featured_image ) ) { ?>
			<div class="featured-image">
			<a class="post-thumbnail" href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
				<?php
				the_post_thumbnail( 'post-thumbnail', array(
						'alt' => the_title_attribute( array(
						'echo' => false,
					) ),
				) );
				?>
			</a>
			</div><!-- featured-image -->
			<?php } ?>

		<?php
		endif; // End is_singular().
	}
endif;

if ( ! function_exists( 'ananyapro_post_categories' ) ) :
	/**
	 * Displays post categories.
	 * @since 1.0.0
	 *
	 */
	function ananyapro_post_categories() {
		// Hide category text for pages.
		if ( 'post' === get_post_type() ) {
			/* translators: used between list items, there is a space after the comma */
			$categories_list = get_the_category_list( esc_html__( ', ', 'ananyapro' ) );
			
			if ( $categories_list ) {
				/* translators: 1: list of categories. */
				printf( '<span class="cat-links"><i class="fas fa-folder-open"></i>&nbsp;' . esc_html( '%1$s' ) . '</span>', $categories_list ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			}
		}
	}
endif;

if ( ! function_exists( 'ananyapro_post_tags' ) ) :
	/**
	 * Displays post tags.
	 * @since 1.0.0
	 *
	 */
	function ananyapro_post_tags() {
		// Hide tag text for pages.
		if ( 'post' === get_post_type() ) {
			/* translators: used between list items, there is a space after the comma */
			$tags_list = get_the_tag_list( '', esc_html_x( ', ', 'list item separator', 'ananyapro' ) );
			if ( $tags_list ) {
				/* translators: 1: list of tags. */
				printf( '<span class="tags-links"><i class="fas fa-tags"></i>' . esc_html( ' %1$s' ) . '</span>', $tags_list ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			}
		}
	}
endif;

if ( ! function_exists( 'ananyapro_post_comments_link' ) ) :
	/**
	 * Displays link to posts comments.
	 * @since 1.0.0
	 *
	 */
	function ananyapro_post_comments_link() {
		if ( ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
			echo '<span class="comments-link">';
			comments_popup_link(
				sprintf(
					wp_kses(
						/* translators: %s: post title */
						__( '0 Comment<span class="screen-reader-text"> on %s</span>', 'ananyapro' ),
						array(
							'span' => array(
								'class' => array(),
							),
						)
					),
					get_the_title()
				)
			);
			echo '</span>';
		}
	}
endif;


if ( ! function_exists( 'ananyapro_post_meta_separator' ) ) :
	/**
	 * Displays post categories.
	 * @since 1.0.0
	 *
	 */
	function ananyapro_post_meta_separator( $sep ) { ?>
		<span class="meta-sep">&nbsp;<?php echo esc_html( $sep ); ?>&nbsp;</span>
	<?php }
endif;

if ( ! function_exists( 'ananyapro_post_navigation' ) ) :
/**
 * Display navigation to next/previous post when applicable.
 *
 * @since 1.0.0
 */
function ananyapro_post_navigation() {
	// Don't print empty markup if there's nowhere to navigate.
	$previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '', true );
	$next     = get_adjacent_post( false, '', false );

	if ( ! $next && ! $previous ) {
		return;
	}

	$previous_post = get_previous_post();
	if ( ! empty( $previous_post ) ) {
		$previous_thumb = get_the_post_thumbnail( $previous_post->ID, 'ananyapro-prevnext-thumbnail' );
	}
	$next_post = get_next_post();
	if ( ! empty( $next_post ) ) {
		$next_thumb     = get_the_post_thumbnail( $next_post->ID, 'ananyapro-prevnext-thumbnail' );
	}

	?>
	<nav class="navigation post-navigation" role="navigation">
		<h2 class="screen-reader-text"><?php esc_html_e( 'Post navigation', 'ananyapro' ); ?></h2>
		<div class="nav-links">
			<?php if ( ! empty( $previous_post ) ) : ?>
				<div class="nav-previous">
					<?php previous_post_link( '%link', ' ' . $previous_thumb . '<div class="nav-innner"><span class="screen-reader-text">' . esc_html_x( 'Previous Post: ', 'post navigation', 'ananyapro' ) . '%title</span><span>' . esc_html__( 'Previous Post', 'ananyapro' ) . '</span> <div>%title</div></div>' ); ?>
				</div>
			<?php endif; ?>
			<?php if ( ! empty( $next_post ) ) : ?>
				<div class="nav-next">
					<?php next_post_link( '%link', '<div class="nav-innner"><span class="screen-reader-text">' . esc_html_x( 'Next Post: ', 'post navigation', 'ananyapro' ) . '%title</span><span>' . esc_html__( 'Next Post', 'ananyapro' ) . '</span><div>%title</div></div>' . $next_thumb . ' ' ); ?>
				</div>
			<?php endif; ?>
		</div><!-- .nav-links -->
	</nav><!-- .navigation-->
	<?php
}
endif;

//display_top_category_setting
if ( ! function_exists( 'ananyapro_get_first_category' ) ) :
	/**
	 * Prints HTML with meta information for the category
	 * @since 1.0.0
	 */
	function ananyapro_get_first_category() {

		$post_top_cat_option = get_theme_mod( 'ananyapro_display_top_category_setting', true );
		// Hide category and tag text for pages.
		if ( 'post' === get_post_type() and $post_top_cat_option === true ) {
			
			$categories = get_the_category();
			if ( ! empty( $categories ) ) {
				 // Get the URL of this category
	    		$category_link = get_category_link( $categories[0]->term_id );
			    printf( '<span class="first-cat"><i class="fas fa-folder-open"></i>&nbsp<a class="top-cat-link" href="'.'%1$s'.'" >' . '%2$s' . '</a></span>', esc_url( $category_link ), esc_html( $categories[0]->name ) );
			}
		}
	}
endif;

if ( ! function_exists( 'ananyapro_single_post_footer' ) ) :
	/**
	 * Prints HTML with meta information for the categories, tags and comments.
	 * @since 1.0.0
	 */
	function ananyapro_single_post_footer() {
		$defaults = ananyapro_get_defaults();
		$hide_single_post_share_buttons = get_theme_mod( 'ananyapro_hide_single_post_share_option', $defaults['hide_single_post_share_option'] );
	?>
		<div class="row single-post-entry-footer">

			<div id="single-cat-links" class="col-md-6 footer-category-links">
			<?php 
			//Display Categories
			ananyapro_post_categories();
			?>
			</div><!-- comments-link -->

			<div id="single-tag-links" class="col-md-6 footer-tag-links">
				<?php
				//Display tags
				ananyapro_post_tags(); ?>
			</div><!-- footer-tag-links -->

		</div><!-- row -->

		<?php if( !$hide_single_post_share_buttons ) { ?>
		<div id="single-social-share" class="social-share">
			<?php echo wp_kses_post( ananyapro_social_sharing_buttons() );  ?>
		</div><!-- single-social-share -->
		<?php } ?>

		<?php	

		//Display edit link
		edit_post_link(
			sprintf(
				wp_kses(
					/* translators: %s: Name of current post. Only visible to screen readers */
					__( 'Edit <span class="screen-reader-text">%s</span>', 'ananyapro' ),
					array(
						'span' => array(
							'class' => array(),
						),
					)
				),
				get_the_title()
			),
			'<span class="edit-link"><i class="fas fa-edit"></i>',
			'</span>'
		);
	}
endif;



if ( ! function_exists ( 'ananyapro_social_sharing_buttons' ) ) { 
	/** 
	 *  Prints HTML for the social share buttons.
	 * @since 1.0.0
	 */
	function ananyapro_social_sharing_buttons() {
		global $post;
		// Show this on post only. if social shared enabled.
		
		// Get current page URL
		$shortURL = esc_url( get_permalink() );
			
		// Get current page title
		$shortTitle = get_the_title();
		$postmediaurl = get_the_post_thumbnail_url($post->id);
		// Construct sharing URL without using any script
		$twitterURL = 'https://twitter.com/share?text='.$shortTitle.'&url='.$shortURL;
		$facebookURL = 'https://www.facebook.com/sharer/sharer.php?u='.$shortURL;
		$linkedInURL = 'https://www.linkedin.com/shareArticle?mini=true&url='.$shortURL.'&title='.$shortTitle;
		$pinterestURL = 'https://pinterest.com/pin/create/button/?url='.$shortURL.'&media='.$postmediaurl.'&description='.$shortTitle;
		$redditURL = 'https://reddit.com/submit?url='.$shortURL.'&title='.$shortTitle;
		$share_msg_str = esc_html__( 'Share: ', 'ananyapro' );
		
		
		// Add sharing button at the end of page/page content
		$content = '<ul>';
		$content .= '<li>'. $share_msg_str .'</li>';

		$content .= '<li><a href="'.esc_url( $facebookURL ).'" target = "_blank"><i class="fab fa-facebook-f"></i></a></li>';
		
		$content .= '<li><a href="'. esc_url( $twitterURL ).'" target = "_blank"><i class="fab fa-twitter"></i></a></li>';
		
		$content .= '<li><a href="'. esc_url( $linkedInURL ) .'" target = "_blank"><i class="fab fa-linkedin"></i></a></li>';
		
		$content .= '<li><a href="'. esc_url( $pinterestURL ) .'" target = "_blank"><i class="fab fa-pinterest-p"></i></a></li>';
		$content .= '<li><a href="'. esc_url( $redditURL ) .'" target = "_blank"><i class="fab fa-reddit"></i></a></li>';
		$content .= '</ul>';
		
		return $content;
		
	}
}

if ( ! function_exists( 'ananyapro_more_link' ) ) :
	/**
	 * Displays the more link on posts.
	 * @since 1.0.0
	 */
	function ananyapro_more_link() {
		$defaults = ananyapro_get_defaults();
		$read_more_text = get_theme_mod( 'ananyapro_read_more_btn_text', $defaults['read_more_btn_text'] );
		$read_more_btn_style = get_theme_mod( 'ananyapro_post_read_more_btn_style', $defaults['read_more_btn_style'] );
		
		if( !empty($read_more_text) ) {
			$read_more_text .= '&nbsp;<i class="fas fa-long-arrow-alt-right"></i>';
			$read_more_markup = '<div class="entry-read-more"><span class="read-more '.esc_attr($read_more_btn_style).'"><a href="'.esc_url( get_permalink() ).'" class="more-link">'.wp_kses_post( $read_more_text ).'</a></span></div>';
		} else {
			$read_more_markup = '...';
		}
		
		return $read_more_markup;
	}
endif;

if ( ! function_exists( 'ananyapro_edit_link' ) ) :
	/**
	 * Returns an accessibility-friendly link to edit a post or page.
	 *
	 * This also gives us a little context about what exactly we're editing
	 * (post or page?) so that users understand a bit more where they are in terms
	 * of the template hierarchy and their content. Helpful when/if the single-page
	 * layout with multiple posts/pages shown gets confusing.
	 */
	function ananyapro_edit_link() {
		edit_post_link(
			sprintf(
				/* translators: %s: Post title. */
				__( 'Edit<span class="screen-reader-text"> "%s"</span>', 'ananyapro' ),
				get_the_title()
			),
			'<span class="edit-link">',
			'</span>'
		);
	}
endif;

if ( ! function_exists( 'wp_body_open' ) ) :
	/**
	 * Fire the wp_body_open action.
	 *
	 * Added for backward compatibility to support pre-5.2.0 WordPress versions.
	 *
	 * @since AnanyaPRO
	 */
	function wp_body_open() {
		/**
		 * Triggered after the opening <body> tag.
		 *
		 * @since AnanyaPRO
		 */
		do_action( 'wp_body_open' );
	}
endif;

function ananyapro_display_widgets( $widget_area_id, $pos ) {

	if( is_customize_preview() ) {
		$sec_preview_class = 'ananyapro-panel';
	} else {
		$sec_preview_class = '';
	} 

	if ( is_active_sidebar( $widget_area_id ) || is_customize_preview() ) { ?>

		<section id="fp-widget-sec-<?php echo esc_attr($pos);?>" class="frontpage-section frontpage-widget-area <?php echo esc_attr($sec_preview_class); ?>">
			<?php if( is_customize_preview() ) { ?>
				<span class="panel ananyapro-panel<?php echo esc_attr($pos ); ?>" id="panel<?php echo esc_attr( $pos ); ?>">
					<span class="ananyapro-panel-title">
					<?php  
					echo esc_html__( 'Widget Area ', 'ananyapro' ); 
					echo esc_attr( $pos);
					if( !is_active_sidebar( $widget_area_id )  ) { ?>
						<p class="widget-placeholder-msg">
							TO ADD WIDGETS GO TO: WIDGETS > FRONTPAGE <?php echo wp_kses_post( $pos ); ?> WIDGET AREA
						</p>
					<?php 
					}
					?>
					</span>
				</span>
				<?php } ?>
			<div class="container">
				<?php dynamic_sidebar( $widget_area_id ); ?>
			</div><!-- container -->
		</section>
	<?php 	
	} 
}

if ( !function_exists( 'ananyapro_display_featured_content_section' ) ) {

	/**
	 * Display Featured Content Section.
	 * @since 1.0.0
	 */

	function ananyapro_display_featured_content_section(  $partial = null, $sec_no = 0  ) {

		if ( is_a( $partial, 'WP_Customize_Partial' ) ) {
			// Find section no and set it up during a selective refresh.
			$sec_no = preg_replace('/[^0-9]/', '', $partial->id );
		}
		$defaults = ananyapro_get_defaults();
		$disable_feat_sec = get_theme_mod( 'ananyapro_disable_feat_sec'.$sec_no, $defaults['disable_feat_sec']);
		$sec_color_scheme = get_theme_mod( 'ananyapro_feat_content_text_color_scheme_sec'.$sec_no, $defaults['section-color-scheme'] );
		$sec_name = get_theme_mod( 'ananyapro_feat_content_section_name_sec'.$sec_no, $defaults['feat_title_sec'] );
		$sec_title = get_theme_mod( 'ananyapro_feat_content_title_sec'.$sec_no,$defaults['feat_title_sec'] );
		$sec_tagline = get_theme_mod( 'ananyapro_feat_content_tagline_sec'.$sec_no,$defaults['feat_tagline_sec'] );
		$sec_image = get_theme_mod( 'ananyapro_feat_image_sec'.$sec_no, $defaults['feat_image_sec'] );
		$sec_image_pos = get_theme_mod('ananyapro_feat_content_image_position'.$sec_no,$defaults['feat_image_position']);
		$sec_content = get_theme_mod( 'ananyapro_feat_content_sec'.$sec_no, $defaults['feat_content_sec'] );
		$sec_button_text = get_theme_mod( 'ananyapro_feat_button_text_sec'.$sec_no, $defaults['feat_text_sec'] );
		$sec_button_url = get_theme_mod( 'ananyapro_feat_button_url_sec'.$sec_no, $defaults['feat_url_sec'] );
		$frontpage_button_style = get_theme_mod( 'ananyapro_frontpage_button_style', $defaults['frontpage_button_style'] );

		if( 'right' === $sec_image_pos ) {
			$col_class = 'col-md-6';
			$col_order = '';
			$sec_item_align = 'items-right';
		} elseif ( 'left' === $sec_image_pos ) {
			$col_class = 'col-md-6';
			$col_order = ' order-md-6';
			$sec_item_align = 'items-left';
		} else {
			$col_class = 'col-md-12';
			//$col_order = '';
			$col_order = ' order-md-12';
			$sec_item_align = 'items-center';
		}?>
		<?php 

		if( is_customize_preview() ) {
			$sec_preview_class = 'ananyapro-panel';
			if( $disable_feat_sec ) {
				$sec_preview_class .= ' disabled';
			}
		} else {
			$sec_preview_class = '';
		}

		if(  ( ! $disable_feat_sec && (!empty( $sec_name ) || !empty( $sec_image ) || !empty( $sec_title ) || !empty( $sec_tagline ) || !empty( $sec_content ) ) ) || is_customize_preview() ) { ?>
		<section id="feat-content-sec-<?php echo esc_attr($sec_no);?>" class="frontpage-section featured-content-section <?php echo esc_attr($sec_item_align); ?> <?php echo esc_attr($sec_preview_class); ?> <?php echo esc_attr($sec_color_scheme); ?>">

				<?php ananyapro_print_customizer_panel_title( esc_html__('Content Section', 'ananyapro' ), $sec_no, $disable_feat_sec ) ?>

				<div class="container">
					<?php if(  ! $disable_feat_sec && ( !empty( $sec_name ) || !empty( $sec_image ) || !empty( $sec_title ) || !empty( $sec_tagline ) || !empty( $sec_content ) || is_customize_preview() ) ) { ?>

					<div class="row featured-sec-wrap">

						
						<!-- Featured content -->
						<div class="<?php echo esc_attr($col_class);  echo esc_attr($col_order); ?> featured-content-wrap my-auto">

							<?php 
							//Display Section Header
							ananyapro_display_frontpage_section_header( $sec_name, $sec_title, $sec_tagline );
							?>

							<?php if( '' != $sec_content  || is_customize_preview() ) { ?>
							<p class="featured-content"><?php echo wp_kses_post( $sec_content ); ?></p>
							<?php } ?>
							<?php 
							if( '' !== $sec_button_text ) {  ?>
							<div class="feat-content-btn-wrap">
							<a href="<?php echo esc_url( $sec_button_url ); ?>" target="_blank"><button class="button feat-sec-btn <?php echo esc_attr( $frontpage_button_style ); ?>"><?php echo esc_html( $sec_button_text ); ?></button></a></div>
							<?php } ?>

						</div><!-- featured-content-wrap -->

						<!-- Featured Image -->
						<?php if( !empty($sec_image)  || is_customize_preview() ) { ?>
							<div class="<?php echo esc_attr($col_class);?> featured-image">
								<?php if( '' != $sec_image || is_customize_preview() ) { ?>
									<img src="<?php echo esc_url($sec_image); ?>" />
								<?php } ?>
							</div><!-- featured-image -->
						<?php } ?>
					</div><!-- row -->
					<?php } ?>
				<?php //} ?>
				</div><!-- container -->
		</section><!-- frontpage-section -->
		<?php }
	} //end of function
} //end of if exists

if ( !function_exists( 'ananyapro_display_featured_category_section' ) ) {

	/**
	 * Display Featured Category Section.
	 * @since 1.0.0
	 */

	function ananyapro_display_featured_category_section(  $partial = null, $sec_no = 0  ) {

		if ( is_a( $partial, 'WP_Customize_Partial' ) ) {
			// Find section no and set it up during a selective refresh.
			$sec_no = preg_replace('/[^0-9]/', '', $partial->id );
		}
		$defaults = ananyapro_get_defaults();
		$disable_feat_sec = get_theme_mod( 'ananyapro_disable_cat_sec_'.$sec_no, $defaults['disable_feat_sec']);
		$sec_color_scheme = get_theme_mod( 'ananyapro_cat_section_text_color_scheme_sec'.$sec_no, $defaults['section-color-scheme'] );
		$sec_name = get_theme_mod( 'ananyapro_cat_section_name_sec'.$sec_no, $defaults['feat_title_sec'] );	
		$sec_title = get_theme_mod( 'ananyapro_cat_title_sec'.$sec_no,$defaults['feat_title_sec'] );
		$sec_tagline = get_theme_mod( 'ananyapro_cat_tagline_sec'.$sec_no,$defaults['feat_tagline_sec'] );
		$category_name = get_theme_mod( 'ananyapro_cat_select_'.$sec_no, $defaults['default_cat'] );
		$sort_order = get_theme_mod( 'ananyapro_cat_sort_order_select_'.$sec_no, $defaults['feat_cat_sort_order'] );
		$layout = get_theme_mod( 'ananyapro_cat_post_layout_select_'.$sec_no, $defaults['feat_cat_layout'] );
		$post_style = get_theme_mod( 'ananyapro_cat_post_style_select_'.$sec_no, $defaults['feat_cat_post_style'] );
		$no_of_posts = get_theme_mod( 'ananyapro_cat_post_per_page_select_'.$sec_no, $defaults['feat_cat_no_of_posts_per_sec'] );
		$cat_sec_btn_text = get_theme_mod( 'ananyapro_cat_sec_button_text_'.$sec_no, $defaults['cat_sec_btn_text'] );
		$hide_post_date = get_theme_mod( 'ananyapro_hide_cat_post_date_'.$sec_no, false );
		$hide_post_excerpt = get_theme_mod( 'ananyapro_hide_cat_sec_post_excerpt_'.$sec_no, false );
		$hide_view_more_link = get_theme_mod( 'ananyapro_hide_cat_sec_view_more_link_'.$sec_no, false );
		$frontpage_button_style = get_theme_mod( 'ananyapro_frontpage_button_style', $defaults['frontpage_button_style'] );
		
		if( $defaults['default_cat'] !== $category_name ) {
			//Number of posts to be displayed in category section
			$cat_query_args = array(
				'posts_per_page' => (int) $no_of_posts,
				'post_status'    => 'publish',
				'category_name'  => $category_name,
				'ignore_sticky_posts' => 1,
				'order'          => 'DESC',
				'orderby'        => $sort_order,
			);
			$cat_posts = new WP_Query( $cat_query_args );
			
			// Get the ID of a given category
		    $category_id = get_cat_ID( $category_name );
		 
		    // Get the URL of this category
		    $category_link = get_category_link( $category_id );
		} 
		
		if( is_customize_preview() ) {
			$sec_preview_class = 'ananyapro-panel';
			if( $disable_feat_sec ) {
				$sec_preview_class .= ' disabled';
			}
		} else {
			$sec_preview_class = '';
		}

		if( ( ! $disable_feat_sec  && ( $defaults['default_cat'] !== $category_name || !empty($sec_name) || !empty($sec_title) || !empty($sec_tagline) ) )|| is_customize_preview()  ) { ?>
			<section id="feat-cat-sec-<?php echo esc_attr($sec_no);?>" class="frontpage-section featured-category-section <?php echo esc_attr($sec_preview_class); ?> <?php echo esc_attr($sec_color_scheme); ?>">

				<?php ananyapro_print_customizer_panel_title( esc_html__('Categorized Posts Section', 'ananyapro' ), $sec_no, $disable_feat_sec ) ?>

				<div class="container">
					<?php if( $defaults['default_cat'] !== $category_name || !empty($sec_name) || !empty($sec_title) || !empty($sec_tagline) || is_customize_preview()  ) { ?>
					<?php 
					//Display Section Header
					if( !$disable_feat_sec  ) {
						ananyapro_display_frontpage_section_header( $sec_name, $sec_title, $sec_tagline, 'items-center' );
					} ?>

					<?php if( $defaults['default_cat'] !== $category_name && ! $disable_feat_sec ) { ?>
						<div class="cat-section-posts-grid <?php echo esc_attr($layout); ?> <?php echo esc_attr($post_style); ?>">
							<?php while ( $cat_posts->have_posts() ) : 
								$cat_posts->the_post(); ?>
							<article id="post-<?php the_ID(); ?>" class="sec-box-shadow">
								<div class="cat-post-featured-image">
									<a href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
										<?php
										
											the_post_thumbnail( 'post-thumbnail', array(
												'alt' => the_title_attribute( array(
												'echo' => false, ) ), ) );
										?>
										<?php
										if( 'style1' === $post_style && !has_post_thumbnail() ) { ?>
											<img width="100%" height="100%" src="<?php echo esc_url(get_parent_theme_file_uri( '/assets/images/blank.jpg' )); ?>">
										<?php } ?>
									</a>
									<?php
										if( 'style1' === $post_style  ) { ?>
									
									<div class="cat-sec-title-over-image">
										<div class="cat-feat-img-overlay"></div>
										<h2 class="cat-post-title"><a href="<?php the_permalink() ?>"><?php the_title(); ?></a></h2>
									</div><!-- cat-sec-title-over-image -->
									<?php } ?>

								</div><!--.cat-post-featured-image-->

								<?php if(  'style2' === $post_style ) : ?>
								<div class="cat-sec-content-wrap">
									<h2 class="cat-post-title"><a href="<?php the_permalink() ?>"><?php the_title(); ?></a></h2>

									<?php if( !$hide_post_date || is_customize_preview() ) {
											ananyapro_posted_on(); 
										}
									?>

									<?php if( !$hide_post_excerpt || is_customize_preview() ) { ?>
										<div class="cat-post-excerpt">
											<?php  
											$read_more_html = '&nbsp;<a href="'.esc_url( get_permalink()).'" class="cat-read-more">...</a>';
		                                        echo wp_kses_post( ananyapro_custom_excerpt( get_the_content(), 20, $read_more_html ) ); 
											
											edit_post_link(
												sprintf(
													/* translators: %s: Name of current post */
													esc_html__( 'Edit %s', 'ananyapro' ),
													the_title( '<span class="screen-reader-text">"', '"</span>', false )
												),
												'<div class="edit-link"><span>',
												'</span></div>'
											);?>
										</div><!-- cat-post-excerpt -->
									<?php } ?>
								</div><!-- cat-sec-content-wrap -->
								<?php endif; ?>
							</article>
							<?php endwhile;
							wp_reset_postdata(); ?>
						</div><!-- row cat-section-posts-grid -->
						
						<?php if( (! $hide_view_more_link && !empty($cat_sec_btn_text) ) ) { ?>
						<div class="cat-view-more">
							<a href="<?php echo esc_url( $category_link ); ?>" class="cat-view-more-link" title="<?php echo esc_attr( $category_name); ?>"><button class="feat-sec-btn <?php echo esc_attr( $frontpage_button_style ); ?>"><?php echo esc_html( $cat_sec_btn_text ); ?>
							</button></a>
						</div><!-- cat-view-more -->
						<?php } 
						}
					} ?>
				</div><!-- container -->
			</section><!-- featured-category-section -->
		<?php }
	} //end of function
} //end of if exists

if ( !function_exists( 'ananyapro_display_CTA_section' ) ) {

	/**
	 * Display Call To Action Section.
	 * @since 1.0.0
	 */

	function ananyapro_display_CTA_section(  $partial = null, $sec_no = 0  ){

		if ( is_a( $partial, 'WP_Customize_Partial' ) ) {
			// Find section no and set it up during a selective refresh.
			$sec_no = preg_replace('/[^0-9]/', '', $partial->id );
		}

		$defaults = ananyapro_get_defaults();
		$disable_cta_sec = get_theme_mod( 'ananyapro_disable_cta_sec'.$sec_no, $defaults['disable_feat_sec']);
		$sec_name = get_theme_mod( 'ananyapro_cta_section_name_sec'.$sec_no, $defaults['feat_title_sec'] );
		$sec_title = get_theme_mod( 'ananyapro_cta_title_sec'.$sec_no, $defaults['feat_title_sec'] );
		$sec_tagline = get_theme_mod( 'ananyapro_cta_tagline_sec'.$sec_no, $defaults['feat_tagline_sec'] );

		$cta_content = get_theme_mod( 'ananyapro_cta_content_sec'.$sec_no , $defaults['feat_content_sec'] );

		$cta_bg_image = get_theme_mod( 'ananyapro_cta_bg_image_sec'.$sec_no, $defaults['feat_image_sec'] );
		$cta_color_scheme = get_theme_mod( 'ananyapro_cta_text_color_scheme_sec'.$sec_no, $defaults['cta-color-scheme'] );
		$cta_bg_color = get_theme_mod( 'ananyapro_cta_bg_color_sec'.$sec_no, $defaults['cta_sec_bg'] );
		$frontpage_button_style = get_theme_mod( 'ananyapro_frontpage_button_style', $defaults['frontpage_button_style'] );

			for( $btn = 1; $btn <=2; $btn++ ){
				$cta_button_text[$btn-1] = get_theme_mod( 'ananyapro_cta_button'.$btn.'_text_sec'.$sec_no, '' );
				$cta_button_url[$btn-1] = get_theme_mod( 'ananyapro_cta_button'.$btn.'_url_sec'.$sec_no, '' );
			}
			$col_class = 'col-md-12';
		?>

		<?php 
			if( is_customize_preview() ) {
				$sec_preview_class = 'ananyapro-panel';
				if( $disable_cta_sec ) {
					$sec_preview_class .= ' disabled';
				}
			} else {
				$sec_preview_class = '';
			}
			$all_text_not_empty = ('' !== $sec_name ) || ('' !== $sec_title ) || ( '' !== $sec_tagline ) || ( '' !== $cta_content ) || ( '' !== $cta_bg_image );

			if( ( ! $disable_cta_sec &&  $all_text_not_empty ) || is_customize_preview() ) { ?>

			<section id="cta-sec-<?php echo esc_attr($sec_no);?>" class="frontpage-section <?php echo esc_attr($sec_preview_class);?> cta-section <?php echo esc_attr($cta_color_scheme); ?>">

				<?php ananyapro_print_customizer_panel_title( esc_html__('CTA Section', 'ananyapro' ), $sec_no, $disable_cta_sec ) ?>

				<div class="container">
					
					<?php if( ! $disable_cta_sec &&  ( ('' !== $sec_name ) || ('' !== $sec_title ) || ( '' !== $sec_tagline ) || ( '' !== $cta_content ) || ( '' !== $cta_bg_image )  || is_customize_preview() )) { ?>
						
					<div class="row featured-cta-wrap">

						<div class="<?php echo esc_attr($col_class); ?> featured-cta my-auto">

							<?php 
							//Display Section Header
							ananyapro_display_frontpage_section_header( $sec_name, $sec_title, $sec_tagline, 'items-center' ) 
							?>
							<div class="cta-content-wrap">

								<?php if( '' != $cta_content || is_customize_preview() ) { ?>
									<p class="cta-content"><?php echo wp_kses_post( $cta_content ); ?></p>
								<?php } ?>

								 <?php if( !empty( $cta_button_text[0] )  || !empty( $cta_button_text[0] ) || is_customize_preview() ) {?>
					            <div class="buttons-wrapper">
					            <?php } ?>

								<?php if( !empty( $cta_button_text[0] )  ) { ?>
									<a href="<?php echo esc_url( $cta_button_url[0] ); ?>" target="_blank"><button class="button feat-sec-btn cta-sec-btn1 <?php echo esc_attr( $frontpage_button_style ); ?>"><?php echo esc_html( $cta_button_text[0] ); ?></button></a>
								<?php } ?>

								<?php if( !empty( $cta_button_text[1] ) ) { ?>
									<a href="<?php echo esc_url( $cta_button_url[1] ); ?>" target="_blank"><button class="button feat-sec-btn cta-sec-btn2 <?php echo esc_attr( $frontpage_button_style ); ?>"><?php echo esc_html( $cta_button_text[1] ); ?></button></a>
								<?php } ?>
							</div><!-- cta-content-wrap -->
							<?php if( !empty( $cta_button_text[0] )  || !empty( $cta_button_text[0] ) || is_customize_preview() ) {?>
					            </div><!-- buttons-wrapper -->
					            <?php } ?>
						</div><!-- featured-cta -->

					</div><!-- row -->
				<?php } ?>
				</div><!-- container -->
				<div class="cta-section-filter"></div><!-- header-filter -->
			</section>
		<?php }
	} //end of function
} //end of if

if ( !function_exists( 'ananyapro_display_featured_pages' ) ) {

	/**
	 * Display Featured Pages Section.
	 * @since 1.0.0
	 */

	function ananyapro_display_featured_pages( $partial = null, $sec_no = 0  ) {
		if ( is_a( $partial, 'WP_Customize_Partial' ) ) {
			// Find section no and set it up during a selective refresh.
			$sec_no = preg_replace('/[^0-9]/', '', $partial->id );
		}
		$defaults = ananyapro_get_defaults();
		$disable_page_sec = get_theme_mod( 'ananyapro_disable_feat_page_sec_'.$sec_no, false );
		$sec_color_scheme = get_theme_mod( 'ananyapro_page_section_text_color_scheme_sec'.$sec_no, $defaults['section-color-scheme'] );
		$col_layout = get_theme_mod( 'ananyapro_pages_layout_sec'.$sec_no, $defaults['pages_layout_sec'] );
		$sec_name = get_theme_mod( 'ananyapro_pages_sec_name_'.$sec_no, $defaults['feat_title_sec'] );
		$sec_title = get_theme_mod( 'ananyapro_pages_sec_title_'.$sec_no, $defaults['feat_title_sec'] );
		$sec_tagline = get_theme_mod( 'ananyapro_pages_sec_tagline_'.$sec_no, $defaults['feat_title_sec'] );
		

		if( is_customize_preview() ) {
			$sec_preview_class = 'ananyapro-panel';
			if( $disable_page_sec ) {
				$sec_preview_class .= ' disabled';
			}
		} else {
			$sec_preview_class = '';
		} 
		?>
		<?php 
		$all_field_not_empty = ('' != $sec_name)  || get_theme_mod( 'ananyapro_feat_page1_sec'.$sec_no ) || get_theme_mod( 'ananyapro_feat_page2_sec'.$sec_no ) || get_theme_mod( 'ananyapro_feat_page3_sec'.$sec_no );
		if( ( ! $disable_page_sec  &&  $all_field_not_empty ) || is_customize_preview() ) { ?>

		<section id="feat-pages-sec-<?php echo esc_attr($sec_no); ?>" class="frontpage-section featured-page-section <?php echo esc_attr($sec_preview_class);?> <?php echo esc_attr($sec_color_scheme); ?>" >

			<?php ananyapro_print_customizer_panel_title( esc_html__('Featured Pages', 'ananyapro' ), $sec_no, $disable_page_sec ) ?>

			<div class="container">
			
			<?php 
				//Display Section Header
				if( !$disable_page_sec  ) {
					ananyapro_display_frontpage_section_header( $sec_name, $sec_title, $sec_tagline, 'items-center' );
				} ?>
			
			<div class="featured-pages-wrapper <?php echo esc_attr($col_layout); ?>">
			<?php
			for ($page = 1; $page <= 3; $page++ ) {
				$featured_page_id  = get_theme_mod( 'ananyapro_feat_page' . $page.'_sec'.$sec_no );
				$page_excerpt = get_theme_mod( 'ananyapro_page'.$page.'_excerpt_sec'.$sec_no, '' );
				$hide_page_title = get_theme_mod( 'ananyapro_hide_page_title_of_page'.$page.'_sec'.$sec_no, false );

				if ( ( !$disable_page_sec &&  $featured_page_id ) || (!$disable_page_sec && is_customize_preview() ) ) { ?>
		
					<article id="feat-page-<?php echo esc_attr($page); ?>-sec-<?php echo esc_attr($sec_no); ?>" class="featured-page" >
							
						<?php if ( $featured_page_id) {
	                   		global $post;
							$post = get_post( $featured_page_id );
							setup_postdata( $post );
							if( is_customize_preview() ) {
								$sec_preview_class = 'ananyapro-panel';
							} else {
								$sec_preview_class = '';
							}?>

							<?php if ( has_post_thumbnail() ) {?>
								<div class="page-sec-feat-image">
									<a class="post-thumbnail" href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
										<?php
										the_post_thumbnail( 'post-thumbnail', array(
												'alt' => the_title_attribute( array(
												'echo' => false,
											) ),
										) );
										?>
									</a>
								</div><!-- .page-sec-feat-image -->
							<?php } ?>

							<div class="page-sec-content-wrap">
								<?php if( ( 'column-1' === $col_layout && !$hide_page_title ) || ( 'column-3' === $col_layout )) { ?>
								<div class="page-title-wrap">
									
									<?php the_title( '<h2 class="feat-page-title"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h2>' );?>
								</div><!-- page-title-wrap -->
								<?php } ?>
								
                                <?php
                                if( 'column-1' === $col_layout )  { ?>
                                	<div class="page-content">
                                	<?php
                                	/* translators: %s: Name of current post */
									the_content( sprintf(
										wp_kses( __( 'Continue reading %s <span class="meta-nav">&rarr;</span>', 'ananyapro' ), array( 'span' => array( 'class' => array() ) ) ),
										the_title( '<span class="screen-reader-text">"', '"</span>', false )
									) ); ?>
									</div><!-- page-content -->
                                <?php } elseif ( 'column-3' === $col_layout ) { ?>
                                	<?php //if( $page_excerpt ) { ?>
	                                	<div class="page-excerpt">
	                                	<?php echo wp_kses_post( $page_excerpt ); ?>
	                                	</div><!-- page-excerpt -->
                                	<?php //} ?>
                                <?php } ?>

								<?php ananyapro_edit_link( get_the_ID() ); ?>
							</div><!-- .page-sec-content-wrap -->
								
						<?php wp_reset_postdata();
						} ?>
					</article><!-- feat-page-section -->
				<?php }  //end of if
			} //end of for?>
			</div><!-- featured-pages-wrapper -->
		</div><!-- container -->
		</section><!-- featured-page-section-pages-wrap -->
		<?php } // end of if
	} //end of function
} //end of if exists


if ( !function_exists( 'ananyapro_display_latest_posts' ) ) {

	/**
	 * Display Latest Posts Section.
	 * @since 1.0.0
	 */

	function ananyapro_display_latest_posts() {
		
	$defaults = ananyapro_get_defaults();
	$disable_latest_post_sec = get_theme_mod( 'ananyapro_disable_latest_post_sec', true );
	$sec_color_scheme = get_theme_mod( 'ananyapro_recent_post_text_color_scheme_sec', $defaults['section-color-scheme'] );
	$sec_name = get_theme_mod( 'ananyapro_recent_post_sec_name', $defaults['feat_title_sec'] );
	$sec_title = get_theme_mod( 'ananyapro_recent_post_sec_title', $defaults['feat_title_sec'] );
	$sec_tagline = get_theme_mod( 'ananyapro_recent_post_sec_tagline', $defaults['feat_title_sec'] );
	$total_no_posts = get_theme_mod( 'ananyapro_recent_posts_total_no', $defaults['latest_post_total_no_posts'] );
	$button_text = get_theme_mod( 'ananyapro_view_blog_button_text', $defaults['latest_post_view_btn_text'] );
	$frontpage_button_style = get_theme_mod( 'ananyapro_frontpage_button_style', $defaults['frontpage_button_style'] );
	
	if( is_customize_preview() ) {
		$sec_preview_class = 'ananyapro-panel';
		if( $disable_latest_post_sec ) {
			$sec_preview_class .= ' disabled';
		}
	} else {
		$sec_preview_class = '';
	}

	 $args = array( // Define WP Query Parameters
        'post_type' => 'post',
        'posts_per_page'=> $total_no_posts,
        'post_status' => 'publish',
        'ignore_sticky_posts' => 1
    );
	$the_query = new WP_Query( $args ); 
	if( have_posts()){ 
		if( ! $disable_latest_post_sec || is_customize_preview() ) { ?>
			<section id="latest-posts-sec" class="frontpage-section latest-posts-section <?php echo esc_attr($sec_preview_class); ?> <?php echo esc_attr($sec_color_scheme); ?>">

				<?php ananyapro_print_customizer_panel_title( esc_html__('Latest Posts', 'ananyapro' ), '', $disable_latest_post_sec ) ?>

				<div class="container">
					
					<?php if( ! $disable_latest_post_sec ) { 
						ananyapro_display_frontpage_section_header( $sec_name, $sec_title, $sec_tagline, 'items-center' );
					
					}//endif ?>

					<?php if( ! $disable_latest_post_sec ) { ?>
					<div class="latest-posts-grid column-3">
						<?php
						while ( $the_query -> have_posts() ) :
							$the_query -> the_post(); ?>
							<article class="post sec-box-shadow">
				        	 	<div class="featured-image">
				                <a class="post-thumbnail" href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
									<?php
									the_post_thumbnail( 'post-thumbnail', array(
											'alt' => the_title_attribute( array(
											'echo' => false,
										) ),
									) );
									?>
								</a>
				            	</div><!-- featured-image -->
				                <div class="latest-posts-title-wrap">
				                <?php the_title( '<h2 class="entry-title"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h2>' ); ?>
				               	<div class="latest-entry-meta">
				                	<?php ananyapro_posted_on(); ?>
				                	<?php ananyapro_get_first_category(); ?>
				                </div>
				            	</div><!-- latest-post-title-wrap -->
				            	<div class="latest-posts-content-wrap">
					            	<div class="entry-excerpt">
					            	<?php $page_read_more_html = '<p class="post-read-more"><a href="'.esc_url( get_permalink()).'" class="post-read-more">'.esc_html__('Read More','ananyapro').'</a></p>';
		                                        echo wp_kses_post( ananyapro_custom_excerpt( get_the_content(), 20, $page_read_more_html ) ); ?>
					            	</div><!-- entry-excerpt -->
				            	</div><!-- latest-post-content-wrap -->
					        </article>
						<?php endwhile; wp_reset_postdata(); //reset?>
				</div><!-- latest-post-grid -->

				<?php
				$blog_page_id = get_option( 'page_for_posts' );
				$button_url   = '#';
				if ( $blog_page_id ) {
					$button_url = get_permalink( $blog_page_id );
				} ?>
				
				<?php if ( !empty( $button_text ) ) : ?>
				<div class="view-blog-wrap">
					<a href="<?php echo esc_url( $button_url ); ?>" title="<?php echo esc_attr( $button_text ); ?>" class="latest-news-button">
						<button class="latest-post-sec-btn feat-sec-btn <?php echo esc_attr( $frontpage_button_style ); ?>">
						<?php echo esc_html( $button_text ); ?>
					</button></a>
				</div><!-- view-blog-wrap -->
				<?php endif; ?>
				<?php }//endif ?>
				</div><!-- container -->
			</section><!-- frontpage-section latest-posts-section -->
    	<?php }
	}

	} //end of function
} //end of if exists

if ( !function_exists( 'ananyapro_display_showcase_categories_section' ) ) {
	/**
	 * Display Categories.
	 * @since 1.0.0
	 */
	function ananyapro_display_showcase_categories_section() {
		$defaults = ananyapro_get_defaults();
		$disable_sec = get_theme_mod( 'ananyapro_disable_showcase_categories_sec', $defaults['disable_feat_sec']);
		
		$sec_name = get_theme_mod( 'ananyapro_showcase_cat_sec_name', $defaults['feat_title_sec'] );	
		$sec_title = get_theme_mod( 'ananyapro_showcase_cat_sec_title', $defaults['feat_title_sec'] );
		$sec_tagline = get_theme_mod( 'ananyapro_showcase_cat_sec_tagline', $defaults['feat_tagline_sec'] );

		for( $c = 1; $c <= 4; $c++ ) {
			$category_name[$c-1] = get_theme_mod( 'ananyapro_showcase_cat_'.$c, $defaults['default_cat'] );

			if( $defaults['default_cat'] !== $category_name[$c-1] ) {
				
				// Get the ID of a given category
			    $category_id[$c-1] = get_cat_ID( $category_name[$c-1] );
			 
			    // Get the URL of this category
			    $category_link[$c-1] = get_category_link( $category_id[$c-1] );
			} else {
				$category_link[$c-1] = '';
			}
			$cat_image[$c-1] = get_theme_mod( 'ananyapro_showcase_cat_image_'.$c, $defaults['feat_image_sec'] );
		}
		
		
		if( is_customize_preview() ) {
			$sec_preview_class = 'ananyapro-panel';
			if( $disable_sec ) {
				$sec_preview_class .= ' disabled';
			}
		} else {
			$sec_preview_class = '';
		}
		$any_cat_selected = ( $defaults['default_cat'] !== $category_name[0] || $defaults['default_cat'] !== $category_name[1] || $defaults['default_cat'] !== $category_name[2] || $defaults['default_cat'] !== $category_name[3] );

		if( ( ! $disable_sec  && ( $any_cat_selected || !empty($sec_name) || !empty($sec_title) || !empty($sec_tagline) ) )|| is_customize_preview()  ) { ?>
			<section id="showcase-categories-section" class="frontpage-section showcase-categories-section <?php echo esc_attr($sec_preview_class); ?>">

				<?php ananyapro_print_customizer_panel_title( esc_html__('Showcase Any 4 Categories', 'ananyapro' ), '', $disable_sec ) ?>

				<div class="container">
					
					<?php if( ! $disable_sec ) { 
						ananyapro_display_frontpage_section_header( $sec_name, $sec_title, $sec_tagline, 'items-left' );
					}//endif ?>
					<div class="showcase-categories-wrap">
						<?php for( $i=0; $i<4; $i++ ) {?>
							<?php 
								if( $defaults['default_cat'] === $category_name[$i] ) {
									$category_name[$i] = '';
								}
							?>
							<div id="showcase-category-<?php echo esc_attr($i+1); ?>"class=" showcase-category-col">
								<div class="category-image">
									<?php if( '' === $cat_image[$i]) {
										$cat_image[$i] = get_parent_theme_file_uri( '/assets/images/blank.jpg' );
									}?>
									<a href="<?php echo esc_url( $category_link[$i]); ?>" target="_blank">
										<img src="<?php echo esc_url($cat_image[$i]); ?>">
									</a>
								</div><!-- category-image -->
								<div class="category-name-wrap">
									<a href="<?php echo esc_url( $category_link[$i]); ?>" target="_blank">
									<h2 class="category-name"><?php echo esc_html( $category_name[$i] ); ?></h2>
									</a>
								</div><!-- category-name -->
							</div><!-- showcase-category-col -->
						<?php } ?>
					</div><!-- showcase-categories-wrap -->
				</div>
			</section>
		<?php }
	}
}

/**
 * Print HTML for Cutomizer Panel Title.
 * @since 1.0.0
 */
function ananyapro_print_customizer_panel_title( $panel_title, $sec_no, $disabled_sec ) {
	if( is_customize_preview() ) { ?>
		<?php if( $disabled_sec ) {
			$class_disabled = 'disabled';
			$dis_msg = ' / Disabled';
		} else {
			$class_disabled = '';
			$dis_msg = '';
		}?>
		<span id="panel<?php echo esc_attr( $sec_no ); ?>" class="panel <?php echo esc_attr( $class_disabled ); ?>" >
			<span class="ananyapro-panel-title">
				<?php 
				echo esc_html( $panel_title ); 
				echo ' ' . esc_attr( $sec_no ). esc_attr( $dis_msg ) ; ?>
			</span>
		</span>
	<?php } 
}

/**
 * Find Sidebar Position for current displaying page.
 * @return string $sidebar_post.
 * @since 1.0.0
 */
function ananyapro_find_sidebar_position() {
	$defaults = ananyapro_get_defaults();
	$static_frontpage_sidebar_pos = esc_attr( get_theme_mod( 'ananyapro_static_front_page_layout', $defaults['static_front_page_layout'] ) );
    $blog_sidebar_pos = esc_attr( get_theme_mod( 'ananyapro_blog_page_layout', $defaults['blog_page_layout'] ) );
    $single_post_sidebar_pos = esc_attr( get_theme_mod( 'ananyapro_single_post_layout', $defaults['single_post_layout'] ) );
    $single_page_sidebar_pos = esc_attr( get_theme_mod( 'ananyapro_single_page_layout', $defaults['single_page_layout'] ) );
    if( is_front_page() && !is_home() ) {
    	$sidebar_pos = $static_frontpage_sidebar_pos;
    } elseif( is_home() || is_archive() || is_search() ){
    	$sidebar_pos = $blog_sidebar_pos;
    } elseif( is_single() ) {
    	$sidebar_pos = $single_post_sidebar_pos;
    } else {
    	$sidebar_pos = $single_page_sidebar_pos;
    }
    return $sidebar_pos;
}

if ( ! function_exists( 'ananyapro_post_list_excerpt_meta' ) ) :
	/**
	 * Prints HTML with meta information for the categories, tags and comments.
	 * @since 1.0
	 */
	function ananyapro_post_list_excerpt_meta() {
		$hide_date = get_theme_mod( 'ananyapro_hide_blog_post_date', false ); 
		$hide_author = get_theme_mod( 'ananyapro_hide_blog_post_author', false );
		$hide_comments_link = get_theme_mod( 'ananyapro_hide_blog_post_comment_link', false );
		$hide_category = get_theme_mod( 'ananyapro_hide_blog_post_category', false );
		?>
		<div class="entry-meta">
			<?php if( !$hide_category ) { ananyapro_get_first_category(); } ?>
			<?php if( !$hide_date ) { 
				ananyapro_posted_on(); } ?>
			<?php if( !$hide_author ) { 
				ananyapro_posted_by(); } ?>
			<?php if( !$hide_comments_link ) { ananyapro_post_comments_link(); } ?>
		</div><!-- entry-meta -->
	<?php
	}
endif;

function ananyapro_change_the_archive_title_prefix( $prefix ) {
	
    if ( is_category() ) {
        $prefix =  __('Category:','ananyapro');
    } elseif( is_tag()) {
    	$prefix = __('Tag:', 'ananyapro');
    } elseif( is_author() ) {
    	$prefix = __('Author:', 'ananyapro');
    } elseif( is_month() ) {
    	$prefix = __('Month:', 'ananyapro');
    } elseif( is_year() ) {
    	$prefix = __('Year:', 'ananyapro');
    } elseif( is_day() ) {
    	$prefix = __('Day:', 'ananyapro');
    }
    
    return $prefix;
}
add_filter( 'get_the_archive_title_prefix', 'ananyapro_change_the_archive_title_prefix' );
