<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package AnanyaPRO
 * @since 1.0.0
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

function ananyapro_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	// Add class if we're viewing the Customizer for easier styling of theme options.
	if ( is_customize_preview() ) {
		$classes[] = 'ananyapro-customizer';
	}
	
	$defaults = ananyapro_get_defaults();
	$enable_frontpage_sections = get_theme_mod( 'ananyapro_enable_frontpage_sections', $defaults['enable_frontpage_sections'] );
	// Add class on front page.
	if ( is_front_page() && 'posts' !== get_option( 'show_on_front' ) && $enable_frontpage_sections ) {
		$classes[] = 'ananyapro-front-page';
	}

	return $classes;
}
add_filter( 'body_class', 'ananyapro_body_classes' );

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 * @since 1.0
 */
function ananyapro_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'ananyapro_pingback_header' );

if ( ! function_exists( 'ananyapro_filter_the_archive_title' ) ) {
	/**
	 * Alter the_archive_title() function to match our original archive title function
	 *
	 * @since 1.0
	 *
	 * @param string $title The archive title
	 * @return string The altered archive title
	 */
	function ananyapro_filter_the_archive_title( $title ) {
		if ( is_category() ) {
			$title = single_cat_title( '', false );
		} elseif ( is_tag() ) {
			$title = single_tag_title( '', false );
		} 

		return $title;

	}
	add_filter( 'get_the_archive_title', 'ananyapro_filter_the_archive_title' );
}
if ( ! function_exists( 'ananyapro_change_the_archive_title_prefix' ) ) :
	/**
	 * Add translation for the_archive_title() prefix
	 *
	 * @since 1.0
	 *
	 * @param string $prefix The archive title prefix
	 * @return string $prefix The translation ready archive title prefix
	 */
	function ananyapro_change_the_archive_title_prefix( $prefix ) {
		
	    if ( is_category() ) {
	        $prefix =  __('Category:','ananyapro');
	    } elseif( is_tag()) {
	    	$prefix = __('Tag:', 'ananyapro');
	    } elseif( is_author() ) {
	    	$prefix = __('Author:', 'ananyapro');
	    } elseif( is_month() ) {
	    	$prefix = __('Month:', 'ananyapro');
	    } elseif( is_year() ) {
	    	$prefix = __('Year:', 'ananyapro');
	    } elseif( is_day() ) {
	    	$prefix = __('Day:', 'ananyapro');
	    }
	    
	    return $prefix;
	}
	add_filter( 'get_the_archive_title_prefix', 'ananyapro_change_the_archive_title_prefix' );
endif;

if ( ! function_exists( 'ananyapro_default_menu' ) ) :
	/**
	 * Display default page as navigation if no custom menu was set
	 */
	function ananyapro_default_menu() {
		echo wp_kses_post( '<div id="navbarResponsive" class="collapse-grp collapse navbar-collapse">' );
		echo wp_kses_post( '<ul id="nav-menu-primary" class="menu nav-menu-primary ml-auto">' . wp_list_pages( 'title_li=&echo=0' ) . '</ul>' );
		echo wp_kses_post( '</div>' );
	}
endif;

if (!function_exists( 'ananyapro_short_excerpt' ) ) {
	/**
	 * Display excerpt of words $length.
	 * @since 1.0
	 */
	function ananyapro_short_excerpt( $length ) {
		$excerpt = explode( ' ', get_the_excerpt(), $length );
		if ( count( $excerpt ) >= $length ) {
			array_pop( $excerpt );
			$excerpt = implode( ' ', $excerpt ).'&nbsp;<a class="read-more-home" href="'. esc_url( get_permalink( get_the_ID() ) ) . '">' . '...' . '<span class="screen-reader-text"> '. esc_html__( 'Read More', 'ananyapro' ).'</span></a>';
		} else {
			$excerpt = implode( ' ', $excerpt );
		}
		$excerpt = preg_replace( '`[[^]]*]`', '', $excerpt );
		echo '<p>'.wp_kses_post( $excerpt ).'</p>';
	}
}

if (!function_exists( 'ananyapro_home_section' ) ) {

    function ananyapro_home_section() {
    	$defaults = ananyapro_get_defaults();
    	$num_sections =  $defaults['total_no_of_each_type_section'];
    	$default_fp_secs = array();
    	
    	for($i=1; $i<=$num_sections; $i++) {
	        array_push( $default_fp_secs,'ananyapro_featured_content_section_'.$i);
	        array_push( $default_fp_secs,'ananyapro_featured_pages_section_'.$i);
	        array_push( $default_fp_secs,'ananyapro_featured_category_section_'.$i);
	        array_push( $default_fp_secs,'ananyapro_CTA_section_'.$i);
	    }
	    array_push( $default_fp_secs, 'ananyapro_latest_posts_section' );
	    array_push( $default_fp_secs, 'ananyapro_showcase_categories_section' );
        $sections = get_theme_mod('ananyapro_frontpage_sections', $default_fp_secs );
        return $sections;
    }
}
if ( !function_exists( 'ananyapro_get_section_position' ) ) {
	function ananyapro_get_section_position( $key ) {
	    $sections = ananyapro_home_section();
	    $position = array_search($key, $sections);
	    $pos = ( $position + 1 ) * 10;
	    return $pos;
	}
}
if ( !function_exists( 'ananyapro_display_frontpage_section_header' ) ) {
	function ananyapro_display_frontpage_section_header( $sec_name, $sec_title, $sec_tagline, $align_class='' ) { ?>
		<div class="section-header <?php echo esc_attr( $align_class ); ?>">
			<?php if( '' != $sec_name  || is_customize_preview() ) { ?>
				<p class="sec-name"><?php echo wp_kses_post($sec_name); ?></p>
			<?php } ?>
			<?php if( '' != $sec_title  || is_customize_preview() ) { ?>
			<h2 class="sec-title"><?php echo wp_kses_post($sec_title); ?></h2>
			<?php } ?>
			<?php if( '' != $sec_tagline  || is_customize_preview() ) { ?>
			<p class="sec-tagline"><?php echo wp_kses_post($sec_tagline); ?></p>
			<?php } ?>
			<?php if( '' != $sec_name || '' != $sec_title  || '' != $sec_tagline ) { ?>
			<div class="sec-divider">
				<div class="sec-divider-internal"></div><!-- sec-divider-internal -->
			</div><!-- sec-divider -->
			<?php } ?>
		</div><!-- section-header -->
	<?php }
}

if( !function_exists( 'ananyapro_display_site_title_tagline' ) ) {
	function ananyapro_display_site_title_tagline() {
		$display_title_tagline = get_theme_mod( 'ananyapro_display_site_title_tagline', true );
		return $display_title_tagline;
	}
}
if( !function_exists( 'ananyapro_print_hero_title' ) ) {
	function ananyapro_print_hero_title() {
		$hero_title = get_theme_mod( 'ananyapro_header_title', '' );
		echo wp_kses_post( $hero_title ); 
	}
}

if( !function_exists( 'ananyapro_print_footer_credit_text' ) ) {
	function ananyapro_print_footer_credit_text() {
		$defaults = ananyapro_get_defaults();
		$copyright = get_theme_mod( 'ananyapro_footer_credit_text', $defaults['footer_credit_text'] );
		$copyright = str_replace( '{year}', date_i18n( 'Y' ), $copyright );
		$copyright = str_replace( '{copy}', '&copy;', $copyright );
		echo wp_kses_post( $copyright ); 
	}
}

if( !function_exists( 'ananyapro_blog_page_header' ) ) {
	function ananyapro_blog_page_header( $blog_page_title, $blog_page_description ) {
		
		if( !empty( $blog_page_title ) || !empty( $blog_page_description )  ) : ?>
			<h2 class="blog-page-title"><span class="blog-title"><?php echo wp_kses_post( $blog_page_title ); ?></span></h2>
			<p class="blog-page-description"><?php echo wp_kses_post( $blog_page_description ); ?></p>
			<?php 
		endif;
	}
}

if( !function_exists( 'ananyapro_archive_page_header' ) ) {
	function ananyapro_archive_page_header() {
		the_archive_title( '<h2 class="entry-title"><span class="archive-title">', '<span></h2>' );
		the_archive_description( '<div class="entry-description">', '</div>' );
	}
}

if( !function_exists( 'ananyapro_single_page_header' ) ) {
	function ananyapro_single_page_header( $single_post_title_position ) {

		the_title( '<h2 class="entry-title">', '</h2>' );
					
		if(  is_single() && 'over-hero-image' === $single_post_title_position ) {
			the_post(); // queue first post
			ananyapro_single_post_meta();
			rewind_posts(); // rewind the loop
		} 
		
	}
}